/***************************************************************************
 *              SMA Solar Technology AG, 34266 Niestetal, Germany
 ***************************************************************************
 *
 *  Copyright (c) 2008-2014
 *  SMA Solar Technology AG
 *  All rights reserved.
 *  This design is confidential and proprietary of SMA Solar Technology AG.
 *
 ***************************************************************************/

#include <string.h>
#include <iostream>
#include <sstream>
#include <unistd.h>
#include "ssdp/SSDPServer.h"
#include "base/Time.h"
#include "base/Logger.h"
#include "base/StringHelper.h"

#define DEFAULT_NOTIFY_INTERVAL   300

#define SERVER_TYPE "Linux/2.6.32 UPnP/1.0 SMA SSDP Server/1.0.0"


SSDPServer::SSDPServer(int httpPort)
{
   _httpPort = httpPort;
   _notifyInterval = DEFAULT_NOTIFY_INTERVAL;
   _forceSendNotify = false;
   _mcSock = new MulticastSocket(SSDP_MCAST_IP, SSDP_MCAST_PORT);
}

SSDPServer::~SSDPServer()
{
   abort();
   sendNotify(false);
   delete _mcSock;
}

void SSDPServer::setNotifyInterval(int notifyInterval)
{
   _notifyInterval = notifyInterval;
}

void SSDPServer::setRootDeviceConfig(const SSDPDeviceConfig &rootDeviceConfig)
{
   _rootDevice = rootDeviceConfig;
}

void SSDPServer::addService(const SSDPServiceConfig &svc)
{
   _services.push_back(svc);
}

std::string SSDPServer::getDeviceDescriptor(const struct in_addr &addr) const
{
   std::stringstream ss;
   ss <<
      "<?xml version=\"1.0\"?>\n"
      "<root xmlns=\"urn:schemas-upnp-org:device-1-0\">\n"
      "\t<specVersion>\n"
      "\t\t<major>1</major>\n"
      "\t\t<minor>0</minor>\n"
      "\t</specVersion>\n"
      "\t<device>\n"
      "\t\t<deviceType>urn:" << _rootDevice.urn << "</deviceType>\n"
      "\t\t<friendlyName>" << _rootDevice.friendlyName << "</friendlyName>\n"
      "\t\t<manufacturer>" << _rootDevice.manufacturer << "</manufacturer>\n"
      "\t\t<manufacturerURL>" << _rootDevice.manufacturerURL << "</manufacturerURL>\n"
      "\t\t<modelDescription>" << _rootDevice.modelDescription << "</modelDescription>\n"
      "\t\t<modelName>" << _rootDevice.modelName << "</modelName>\n"
      "\t\t<modelNumber>" << _rootDevice.modelNumber << "</modelNumber>\n"
      "\t\t<modelURL>" << _rootDevice.modelURL << "</modelURL>\n"
      "\t\t<serialNumber>" << _rootDevice.serialNumber << "</serialNumber>\n"
      "\t\t<UDN>uuid:" << _rootDevice.uuid << "</UDN>\n"
      "\t\t<serviceList>\n";
   for (std::vector<SSDPServiceConfig>::const_iterator svcIt = _services.begin(); svcIt != _services.end(); ++svcIt)
   {
      ss <<
      "\t\t\t<service>\n"
      "\t\t\t\t<serviceType>urn:" << svcIt->typeUrn << ":service:NULL:1</serviceType>\n"
      "\t\t\t\t<serviceId>urn:" << svcIt->idUrn << ":serviceId:NULL</serviceId>\n"
      "\t\t\t\t<SCPDURL>" << svcIt->SCPDURL << "</SCPDURL>\n"
      "\t\t\t\t<controlURL>" << svcIt->controlURL << "</controlURL>\n"
      "\t\t\t\t<eventSubURL>" << svcIt->eventSubURL << "</eventSubURL>\n"
      "\t\t\t</service>\n";
   }
   ss <<
      "\t\t</serviceList>\n"
      "\t\t<presentationURL>" << _rootDevice.presentationURL << "</presentationURL>\n";

   struct in_addr localAddr;
   findLocalInterface(addr, localAddr);

   for (std::vector<std::string>::const_iterator addElIt = _rootDevice.additionalElements.begin();
         addElIt != _rootDevice.additionalElements.end(); ++addElIt)
   {
      std::string element = *addElIt;
      StringHelper::replace(element, "@IFACE_ADDR@", inet_ntoa(localAddr), true);
      ss << "\t\t" << element << "\n";
   }
   ss <<
      "\t</device>\n"
      "</root>\n";
   return ss.str();
}

std::string SSDPServer::getDiscoveryNT_USN(SSDPDiscoveryType type, bool useSearchTarget) const
{
   const std::string targetPrefix = (useSearchTarget ? "ST" : "NT");

   switch (type)
   {
   case SSDP_DT_ROOTDEVICE: return
      targetPrefix + ": upnp:rootdevice\r\n" \
      "USN: uuid:" + _rootDevice.uuid + "::upnp:rootdevice\r\n";
   case SSDP_DT_DEVICE_TYPE: return
      targetPrefix + ": urn:" + _rootDevice.urn + "\r\n" \
      "USN: uuid:" + _rootDevice.uuid + "::urn:" + _rootDevice.urn + "\r\n";
   case SSDP_DT_DEVICEID_TYPE: return
      targetPrefix + ": uuid:" + _rootDevice.uuid + "\r\n" \
      "USN: uuid:" + _rootDevice.uuid + "\r\n";
   default:
      return "";
   }
}

std::string SSDPServer::getDiscoveryServiceNT_USN(const SSDPServiceConfig &svc, bool useSearchTarget) const
{
   const std::string targetPrefix = (useSearchTarget ? "ST" : "NT");
   return targetPrefix + ": urn:" + svc.typeUrn + "\r\n" \
         "USN: uuid:" + _rootDevice.uuid + "::urn:" + svc.typeUrn + "\r\n";
}

void SSDPServer::sendNotify(bool alive)
{
   // multicast notify message on every interface
   std::vector<network_iface_t> ifaces = _mcSock->getInterfaces();
   for (std::vector<network_iface_t>::iterator it = ifaces.begin(); it != ifaces.end(); ++it)
   {
      std::stringstream header;
      header <<
         "NOTIFY * HTTP/1.1\r\n" \
         "HOST: "SSDP_MCAST_IP":"STR(SSDP_MCAST_PORT)"\r\n";
      if (alive)
      {
         header <<
            "CACHE-CONTROL: max-age = 1800\r\n" \
            "SERVER: "SERVER_TYPE"\r\n" \
            "NTS: ssdp:alive\r\n" \
            "LOCATION: http://" << inet_ntoa(it->addr) << ":" << _httpPort << _rootDevice.locationPath << "\r\n";
      }
      else
      {
         header <<
            "NTS: ssdp:byebye\r\n";
      }

      _mcSock->selectInterface(it->addr.s_addr);

      std::string msg;
      msg = header.str() + getDiscoveryNT_USN(SSDP_DT_ROOTDEVICE, false) + "\r\n";
      if (_mcSock->send(msg.c_str(), msg.size()) == -1)
         Logger::log() << "Failed to send SSDP notify" << std::endl;

      msg = header.str() + getDiscoveryNT_USN(SSDP_DT_DEVICEID_TYPE, false) + "\r\n";
      if (_mcSock->send(msg.c_str(), msg.size()) == -1)
         Logger::log() << "Failed to send SSDP notify" << std::endl;

      msg = header.str() + getDiscoveryNT_USN(SSDP_DT_DEVICE_TYPE, false) + "\r\n";
      if (_mcSock->send(msg.c_str(), msg.size()) == -1)
         Logger::log() << "Failed to send SSDP notify" << std::endl;

      for (std::vector<SSDPServiceConfig>::iterator confIt = _services.begin(); confIt != _services.end(); ++confIt)
      {
         msg = header.str() + getDiscoveryServiceNT_USN(*confIt, true) + "\r\n";
         if (_mcSock->send(msg.c_str(), msg.size()) == -1)
            Logger::log() << "Failed to send SSDP notify" << std::endl;
      }
   }
}

void SSDPServer::sendMSearchResponse(const std::string &searchTarget, const struct in_addr &ifaceAddr, const struct sockaddr_in &dstAddr)
{
   // build M-SEARCH response

   const std::string ascTime = Time::toASCIITime(Time::getCurrentTime());
   std::stringstream header;
   header <<
      "HTTP/1.1 200 OK\r\n"
      "CACHE-CONTROL: max-age = 1800\r\n"
      "DATE: " << ascTime << "\r\n"
      "EXT: \r\n" \
      "LOCATION: http://" << inet_ntoa(ifaceAddr) << ":" << _httpPort << _rootDevice.locationPath << "\r\n" \
      "SERVER: "SERVER_TYPE"\r\n";

   // send M-SEARCH response

   std::string response;
   if (searchTarget == "ssdp:all")
   {
      response = header.str() + getDiscoveryNT_USN(SSDP_DT_ROOTDEVICE, true) + "\r\n";
      if (_mcSock->send(response.c_str(), response.size(), dstAddr) == -1)
         Logger::log() << "Failed to send SSDP response" << std::endl;

      response = header.str() + getDiscoveryNT_USN(SSDP_DT_DEVICEID_TYPE, true) + "\r\n";
      if (_mcSock->send(response.c_str(), response.size(), dstAddr) == -1)
         Logger::log() << "Failed to send SSDP response" << std::endl;

      response = header.str() + getDiscoveryNT_USN(SSDP_DT_DEVICE_TYPE, true) + "\r\n";
      if (_mcSock->send(response.c_str(), response.size(), dstAddr) == -1)
         Logger::log() << "Failed to send SSDP response" << std::endl;

      for (std::vector<SSDPServiceConfig>::iterator it = _services.begin(); it != _services.end(); ++it) {
         response = header.str() + getDiscoveryServiceNT_USN(*it, true) + "\r\n";
         if (_mcSock->send(response.c_str(), response.size(), dstAddr) == -1)
            Logger::log() << "Failed to send SSDP response" << std::endl;
      }
   }
   else if (searchTarget == "upnp:rootdevice")
   {
      response = header.str() + getDiscoveryNT_USN(SSDP_DT_ROOTDEVICE, true) + "\r\n";
      if (_mcSock->send(response.c_str(), response.size(), dstAddr) == -1)
         Logger::log() << "Failed to send SSDP response" << std::endl;
   }
   else if (searchTarget == "uuid:" + _rootDevice.uuid)
   {
      response = header.str() + getDiscoveryNT_USN(SSDP_DT_DEVICEID_TYPE, true) + "\r\n";
      if (_mcSock->send(response.c_str(), response.size(), dstAddr) == -1)
         Logger::log() << "Failed to send SSDP response" << std::endl;
   }
   else if (searchTarget == "urn:" + _rootDevice.urn)
   {
      response = header.str() + getDiscoveryNT_USN(SSDP_DT_DEVICE_TYPE, true) + "\r\n";
      if (_mcSock->send(response.c_str(), response.size(), dstAddr) == -1)
         Logger::log() << "Failed to send SSDP response" << std::endl;
   }
   else
   {
      for (std::vector<SSDPServiceConfig>::iterator it = _services.begin(); it != _services.end(); ++it)
      {
         if (searchTarget == "urn:" + it->typeUrn)
         {
            response = header.str() + getDiscoveryServiceNT_USN(*it, true) + "\r\n";
            if (_mcSock->send(response.c_str(), response.size(), dstAddr) == -1)
               Logger::log() << "Failed to send SSDP response" << std::endl;
            break;
         }
      }
   }
}

bool SSDPServer::findLocalInterface(const struct in_addr &remoteAddr, struct in_addr &localAddr) const
{
   localAddr.s_addr = 0;
   std::vector<network_iface_t> ifaces = _mcSock->getInterfaces();
   for (std::vector<network_iface_t>::iterator it = ifaces.begin(); it != ifaces.end(); ++it)
   {
      if ((it->addr.s_addr & it->netmask) == (remoteAddr.s_addr & it->netmask)) {
         localAddr = it->addr;
         return true;
      }
   }
   return false;
}

void SSDPServer::onMulticastReceived(const char *buffer, ssize_t bufSize, const struct sockaddr_in &recAddr)
{
   std::string msg(buffer, bufSize);

   // accept only M-SEARCH requests with type "MAN: ssdp:discover"

   if (!StringHelper::startsWith(msg, "M-SEARCH * HTTP/1.1"))
      return;

   if (msg.find("ssdp:discover") == std::string::npos)
      return;

   // extract search-target (ST) field from M-SEARCH request

   std::vector<std::string> lines;
   StringHelper::split(msg, "\r\n", lines, true);
   std::string searchTarget;
   for (size_t i = 1; i < lines.size(); ++i) {
      if (StringHelper::startsWith(lines[i], "ST:"))
      {
         searchTarget = StringHelper::strip(lines[i].c_str() + 3);
      }
   }


   // find suitable network interface for M_SEARCH request
   struct in_addr ifaceAddr;
   if (!findLocalInterface(recAddr.sin_addr, ifaceAddr))
   {
      Logger::log() << "No iface found for mcast-request from " << inet_ntoa(recAddr.sin_addr) << std::endl;
      return;
   }

   // send as unicast message (according to UPnP v1.0 SSDP spec)
   sendMSearchResponse(searchTarget, ifaceAddr, recAddr);
}

void SSDPServer::onInterfaceChanged()
{
   // re-send notify messages
   _forceSendNotify = true;
   Logger::log() << "SSDPServer: Change in Network Adapters -> NOTIFY triggered!" << std::endl;
}

void SSDPServer::run()
{
   _mcSock->listen(this);

   timestamp_t nextNotifyTime = 0;
   while (!isAborted()) {
      if (Time::getCurrentTime() > nextNotifyTime || _forceSendNotify) {
         sendNotify(true);
         nextNotifyTime = Time::getCurrentTime() + _notifyInterval;
         _forceSendNotify = false;
      }
      usleep(200000);
   }
}
