/***************************************************************************
 *              SMA Solar Technology AG, 34266 Niestetal, Germany
 ***************************************************************************
 *
 *  Copyright (c) 2008-2014
 *  SMA Solar Technology AG
 *  All rights reserved.
 *  This design is confidential and proprietary of SMA Solar Technology AG.
 *
 ***************************************************************************/

#ifndef SSDPDISCOVERY_H_
#define SSDPDISCOVERY_H_

#include <set>
#include "base/Thread.h"
#include "base/MulticastSocket.h"
#include "SSDPCommon.h"

enum SSDPDeviceState
{
   STATE_ALIVE,
   STATE_BYEBYE
};

typedef std::string ssdp_uuid_t;

/*! @class SSDPDeviceInfo
 *
 * @brief Encapsulates the information of a UPnP device found by the SSDP discovery process.
 */
class SSDPDeviceInfo
{
public:
   static const int TIMEOUT_UNDEFINED = -1;

   SSDPDeviceInfo() :
      uuid(),
      location(),
      timeout(TIMEOUT_UNDEFINED)
   {
      memset(&devAddr, 0, sizeof(devAddr));
   }

   SSDPDeviceInfo(const ssdp_uuid_t &uuid_, const std::string &location_, int timeout_, const struct sockaddr_in &devAddr_) :
      uuid(uuid_),
      location(location_),
      timeout(timeout_),
      devAddr(devAddr_) {}

   /*!
    * @brief UPnP unique device ID
    */
   ssdp_uuid_t uuid;

   /*!
    * @brief URL for presentation webpage, etc.
    */
   std::string location;

   /*!
    * @brief Device timeout in seconds (derived from Cache-Control element)
    * A device can be considered as not available anymore if no new SSDPDeviceInfo is received in this time.
    * If no Cache-Control element was set (although it is required in SSDP) the value of this field must be TIMEOUT_UNDEFINED.
    */
   int timeout;

   /*!
    * @brief Device IP-address
    */
   struct sockaddr_in devAddr;
};

/*! @class ISSDPDiscoveryHandler
 *
 * @brief This interface provides a routine which is called when a new device was found by the SSDP discovery.
 */
class ISSDPDiscoveryHandler
{
public:
   /*!
    * @brief Called when a new device was found by the SSDP discovery.
    */
   virtual void updateSSDPDevice(SSDPDeviceState state, const SSDPDeviceInfo &info) = 0;

protected:
   virtual ~ISSDPDiscoveryHandler() {}
};

/*! @class SSDPDiscovery
 *
 * @brief Searches for new UPnP devices using the SSDP protocol.
 */
class SSDPDiscovery : public Thread, private IMulticastListener
{
public:
   /*!
    * @brief Constructor.
    * @param handler the handler to notify when a UPnP device is found or is not available anymore.
    */
   SSDPDiscovery(ISSDPDiscoveryHandler *handler);

   /*!
    * @brief Destructor.
    */
   ~SSDPDiscovery();

   void setDiscoveryPeriod(int period);

   void addSearchTarget(const std::string &target);

   /*!
    * @brief Sets the verbosity of console output.
    */
   void setVerbosity(int level);

   /*!
    * @see Thread.run()
    */
   virtual void run();

private:
   /*!
    * @brief the verbosity used for console output
    */
   int _verbosity;

   /*!
    * @brief the handler associated with this discovery
    */
   ISSDPDiscoveryHandler *_handler;

   /*!
    * @brief the socket used to receive NOTIFY (multicast) messages.
    */
   MulticastSocket *_notifyMcastSock;

   /*!
    * @brief the socket used to send M-SEARCH (multicast) requests and receive M-SEARCH (unicast) responses.
    */
   MulticastSocket *_msearchMcastSock;

   std::set<std::string> _searchTargets;

   /*!
    * @brief regular interval between two M-Search request messages
    */
   int _discoveryPeriod;

   /*!
    * @brief if true, forces immediate sending of M-Search request messages.
    *
    * This resets the regular wait interval, so that the next message after the forced one
    * will be sent after the regular interval.
    */
   bool _forceDiscovery;

   void sendMSearchRequests();

   /*!
    * @brief Called when a new multicast packet was received by the multicast socket.
    */
   virtual void onMulticastReceived(const char *buffer, ssize_t bufSize, const struct sockaddr_in &recAddr);

   virtual void onInterfaceChanged();
};

#endif
