/***************************************************************************
 *              SMA Solar Technology AG, 34266 Niestetal, Germany
 ***************************************************************************
 *
 *  Copyright (c) 2008-2014
 *  SMA Solar Technology AG
 *  All rights reserved.
 *  This design is confidential and proprietary of SMA Solar Technology AG.
 *
 ***************************************************************************/

#include <string>
#include <string.h>
#include <sstream>
#include <stdlib.h>
#include <unistd.h>
#include "ssdp/SSDPDiscovery.h"
#include "base/StringHelper.h"
#include "base/Logger.h"
#include "base/Console.h"
#include "base/Time.h"

#define DEFAULT_DISCOVERY_PERIOD 300

SSDPDiscovery::SSDPDiscovery(ISSDPDiscoveryHandler *handler) :
   _verbosity(0),
   _handler(handler),
   _notifyMcastSock(0),
   _msearchMcastSock(0),
   _discoveryPeriod(DEFAULT_DISCOVERY_PERIOD),
   _forceDiscovery(false) {}

SSDPDiscovery::~SSDPDiscovery()
{
   abort();
   delete _notifyMcastSock;
   delete _msearchMcastSock;
}

void SSDPDiscovery::addSearchTarget(const std::string &target)
{
   _searchTargets.insert(target);
}

void SSDPDiscovery::setDiscoveryPeriod(int period)
{
   _discoveryPeriod = period;
}

void SSDPDiscovery::onMulticastReceived(const char *buffer, ssize_t bufSize, const struct sockaddr_in &recAddr)
{
   // Note: this handler is used by two multicast sockets.
   // No data locking is needed, as common data is only read but not written.

   std::string msg(buffer, bufSize);
   bool isNotify = StringHelper::startsWith(msg, "NOTIFY * HTTP/1.1");
   bool isMSearchResponse = StringHelper::startsWith(msg, "HTTP/1.1 200 OK");

   if (!isNotify && !isMSearchResponse)
      return;

   std::vector<std::string> lines;
   StringHelper::split(msg, "\r\n", lines, true);

   std::string notificationType = "";
   std::string notificationSubType = "";
   std::string location = "";
   int cacheControl = SSDPDeviceInfo::TIMEOUT_UNDEFINED;
   ssdp_uuid_t uuid = "";

   for (size_t i = 1; i < lines.size(); ++i) {
      const std::string line = lines[i];
      if (StringHelper::startsWith(line, "NT:"))
         notificationType = StringHelper::strip(line.c_str() + strlen("NT:"));
      else if (StringHelper::startsWith(line, "ST:"))
         notificationType = StringHelper::strip(line.c_str() + strlen("ST:"));
      else if (StringHelper::startsWith(line, "NTS:"))
         notificationSubType = StringHelper::strip(line.c_str() + strlen("NTS:"));
      else if (StringHelper::startsWith(line, "LOCATION:"))
         location = StringHelper::strip(line.c_str() + strlen("LOCATION:"));
      else if (StringHelper::startsWith(line, "CACHE-CONTROL:"))
         cacheControl = atoi(line.c_str() + strlen("CACHE-CONTROL:"));
      else if (StringHelper::startsWith(line, "USN:"))
      {
         size_t uuidStartPos = line.find("uuid:");
         if (uuidStartPos == std::string::npos)
         {
            Logger::log() << "SSDPDiscovery: received SSDP message with invalid USN field (no UUID)\n" << std::endl;
            return;
         }

         uuidStartPos += strlen("uuid:");
         size_t uuidEndPos = line.find("::urn", uuidStartPos);
         if (uuidEndPos != std::string::npos)
            uuid = line.substr(uuidStartPos, uuidEndPos - uuidStartPos);
         else
            uuid = line.substr(uuidStartPos);
      }
   }

   if (_searchTargets.find(notificationType) != _searchTargets.end())
   {
      bool advertise = true;
      if (isNotify) {
         if (notificationSubType == "ssdp:alive")
            advertise = true;
         else if (notificationSubType == "ssdp:byebye")
            advertise = false;
         else
            return;
      }

      if (_verbosity >= 1)
      {
         if (isNotify)
            Logger::log() << "SSDPDiscovery: received SSDP NOTIFY from Gateway:\n" << TMAGENTA(msg) << std::endl;

         if (isMSearchResponse)
            Logger::log() << "SSDPDiscovery: received SSDP M-SEARCH response from Gateway:\n" << TMAGENTA(msg) << std::endl;
      }

      // lock to assure that the handler is only called once at a time
      // (as this handler is used by two multicast sockets)
      setDataLock();

      if (advertise)
      {
         SSDPDeviceInfo info(uuid, location, cacheControl, recAddr);
         _handler->updateSSDPDevice(STATE_ALIVE, info);
      }
      else
      {
         SSDPDeviceInfo info(uuid, "", 0, recAddr);
         _handler->updateSSDPDevice(STATE_BYEBYE, info);
      }

      unsetDataLock();
   }

}

void SSDPDiscovery::onInterfaceChanged()
{
   _forceDiscovery = true;
   Logger::log() << "SSDPDiscovery: Change in Network Adapters -> M-Search triggered!" << std::endl;
}

void SSDPDiscovery::setVerbosity(int level)
{
   _verbosity = level;
}

void SSDPDiscovery::sendMSearchRequests()
{
   std::vector<network_iface_t> ifaces = _msearchMcastSock->getInterfaces();
   for (std::vector<network_iface_t>::iterator ifIt = ifaces.begin(); ifIt != ifaces.end(); ++ifIt)
   {
      for (std::set<std::string>::iterator stIt = _searchTargets.begin(); stIt != _searchTargets.end(); ++stIt)
      {
         std::stringstream ss;
         ss <<
            "M-SEARCH * HTTP/1.1\r\n"
            "HOST: "SSDP_MCAST_IP":"STR(SSDP_MCAST_PORT)"\r\n"
            "MAN: \"ssdp:discover\"\r\n"
            "MX: 10\r\n"
            "ST: " << *stIt << "\r\n"
            "\r\n";
         _msearchMcastSock->selectInterface(ifIt->addr.s_addr);

         std::string msg = ss.str();

         if (_verbosity >= 1)
         {
            Logger::log() << "SSDPDiscovery: sending M-SEARCH request to Gateway\n" << TCYAN(msg) << std::endl;
         }

         if (_msearchMcastSock->send(msg.c_str(), msg.size()) != (ssize_t)msg.size())
         {
            Logger::log() << "Failed to send M-SEARCH message" << std::endl;
         }
      }
   }
}

void SSDPDiscovery::run()
{
   _notifyMcastSock = new MulticastSocket(SSDP_MCAST_IP, SSDP_MCAST_PORT, false);
   _notifyMcastSock->listen(this);

   _msearchMcastSock = new MulticastSocket(SSDP_MCAST_IP, SSDP_MCAST_PORT, true);
   _msearchMcastSock->listen(this);

   timestamp_t nextDiscoveryTime = 0;
   while (!isAborted()) {
      if (Time::getCurrentTime() > nextDiscoveryTime || _forceDiscovery) {
         sendMSearchRequests();
         nextDiscoveryTime = Time::getCurrentTime() + _discoveryPeriod;
         _forceDiscovery = false;
      }
      usleep(2000000);
   }

   delete _notifyMcastSock;
   _notifyMcastSock = 0;

   delete _msearchMcastSock;
   _msearchMcastSock = 0;
}
