/***************************************************************************
 *              SMA Solar Technology AG, 34266 Niestetal, Germany
 ***************************************************************************
 *
 *  Copyright (c) 2008-2014
 *  SMA Solar Technology AG
 *  All rights reserved.
 *  This design is confidential and proprietary of SMA Solar Technology AG.
 *
 ***************************************************************************/

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sstream>
#include <string.h>
#include <stdio.h>
#include <base/Logger.h>
#include <errno.h>
#include "HTTPCommon.h"
#include "base/StringHelper.h"

#define BUF_SIZ 4096

const char* HTTPResponseField::CONTENT_TYPE = "Content-Type";

static ssize_t recv_timeout(int sockfd, void *buf, size_t len, int flags, struct timeval *timeout)
{
   fd_set fdset;
   FD_ZERO(&fdset);
   FD_SET(sockfd, &fdset);
   if (select(sockfd + 1, &fdset, NULL, NULL, timeout) == -1)
      return -1;

   if (FD_ISSET(sockfd, &fdset)) {
      return recv(sockfd, buf, len, 0);
   }

   errno = ETIMEDOUT;
   return -1;
}

bool HTTPCommon::recvMessage(int sock, bool client, std::string &header, std::string &body, time_t timeout)
{
   int bytes;
   int contentLen = 0;
   char buffer[BUF_SIZ];
   bool inHeader = true;
   bool hasContentLen = false;

   // a HTTP server cannot detect the end of a message w/o a CONTENT-LENGTH (if the chunked mode is not used)
   // as the socket is not closed after the request. As a GET-Request does not a CONTENT-LENGTH we will just
   // assume length 0 instead of returning an error.
   bool requireContentLength = true; //!client;

   struct timeval tv;
   tv.tv_sec = timeout;
   tv.tv_usec = 0;

   std::stringstream ss;
   while ((bytes = recv_timeout(sock, buffer, sizeof(buffer), 0, &tv)) > 0){
      std::string chunk(buffer, bytes);
      ss.write(chunk.c_str(), chunk.size());

      if (inHeader)
      {
         size_t headerEnd = chunk.find("\r\n\r\n");
         if(headerEnd != std::string::npos)
         {
            // extract header
            header = ss.str().substr(0, headerEnd + 2);
            // save already received body contents
            std::string bodyBuf = ss.str().substr(headerEnd + 4);
            // reset buffer (do not initialize with bodyBuf, the buffer cannot be expanded with write() or << for some reason)
            ss.str("");
            // and append body contents
            ss.write(bodyBuf.c_str(), bodyBuf.size());

            // try to get content length
            const char *contentLenStr = strcasestr(header.c_str(), "content-length");
            if (contentLenStr)
            {
               contentLenStr += strlen("content-length");
               sscanf(contentLenStr, ":%d", &contentLen);
               hasContentLen = true;
            } else {
               if (requireContentLength) {
                  hasContentLen = true;
                  contentLen = 0;
               } else {
                  hasContentLen = false;
               }
            }

            if (hasContentLen && contentLen == 0)
            {
               body = "";
               return true;
            }

            inHeader = false;
         }
      }

      if (!inHeader && hasContentLen && ss.str().size() >= (size_t)contentLen)
      {
         body = std::string(ss.str(), 0, contentLen);
         return true;
      }
   }

   if (!inHeader && !hasContentLen) {
      body = ss.str();
      return true;
   }

   return false;
}

bool HTTPMessageHeaderBuffer::getField(const std::string &name, std::string &value) const
{
   // start with the second line as the first line contains the status
   size_t lineStartPos = buffer.find("\r\n");
   if (lineStartPos == std::string::npos)
      return false;
   lineStartPos += 2;

   size_t lineEndPos;
   while ((lineEndPos = buffer.find("\r\n", lineStartPos)) != std::string::npos) {
      size_t sepPos = buffer.find(':', lineStartPos);
      if (sepPos == std::string::npos)
         break; // no further fields in buffer
      if (sepPos < lineEndPos) {
         const std::string nameFound = buffer.substr(lineStartPos, sepPos - lineStartPos);
         // field names are case-insensitive
         if (StringHelper::compareIgnoreCase(name, nameFound) == 0) {
            value = StringHelper::strip(buffer.substr(sepPos+1, lineEndPos - sepPos - 1));
            return true;
         }
      }

      lineStartPos = lineEndPos + 2;
   }
   return false;
}
