#ifndef COMMONGATEWAYHANDLER_H_
#define COMMONGATEWAYHANDLER_H_

#include "../GatewaySimulator.h"
#include "base/Time.h"
#include <cmath>
#include "structs/DeviceTypeRef.h"
#include "structs/PowerMeasurementMethodRef.h"
#include "structs/StatusRef.h"

/*
 * IF SEND_CAN_FRAME is enabled, a so called "Can"-Timeframe is sent instead of a "Must"-Timeframe.
 * "Must" means that a device has to be switched on for a specific amount of time. A "CAN"-Frame is
 * optional. The device might only run if energy is cheap, depending on the constraint selected by
 * EcoLimit or PriceLevel.
 * Note that Can-Frames always have a MinRunningTime=0 (or MinEnergy=0) whereas Must-Frames have
 * MinRunningTime=MaxRunningTime.
 * Timeframes with MinRunningTime!=MaxRunningTime (and MinRunningTime!=0) (called Can+Must Frames)
 * are not supported at the moment.
 */
//#define SEND_CAN_FRAME

class CommonGatewayHandler : public IGatewayHandler
{
public:
   CommonGatewayHandler() :
      _startTime(-1),
      _statusOn(false)
   {}

   virtual std::vector<std::string> getDeviceIDs()
   {
      std::vector<std::string> devices;
      devices.push_back("F-11223344-112233445566-00");
      return devices;
   }

   /*!
    * @brief Called whenever the Home Manager polls this gateway (once per minute) to retrieve device and planning information.
    */
   virtual void getDeviceMsg(msg_type_t type, SEMP::Device2EM &msg, const std::string &id)
   {
      switch (type)
      {
      case MSG_DEVICE_INFO: {
         SEMP::DeviceInfo devInfo;
         devInfo.getIdentification().setDeviceId(id);
         devInfo.getIdentification().setDeviceName("Test Device");
         devInfo.getIdentification().setDeviceVendor("Vendor");
         devInfo.getIdentification().setDeviceType(SEMP::DeviceTypeRef::Other);
         devInfo.getIdentification().setDeviceSerial("ART34-2123-XYZ");
         devInfo.getCharacteristics().setMaxPowerConsumption(1500);
         devInfo.getCharacteristics().setMinOffTime(600);
         devInfo.getCharacteristics().setMinOnTime(1200);
         devInfo.getCapabilities().getCurrentPower().setMethod(SEMP::PowerMeasurementMethodRef::Measurement);
         devInfo.getCapabilities().getTimestamps().setAbsoluteTimestamps(false); // use relative timestamps (0 = now)
         devInfo.getCapabilities().getInterruptions().setInterruptionsAllowed(true);
#ifdef SEND_CAN_FRAME
         devInfo.getCapabilities().getRequests().setOptionalEnergy(true);
#else
         devInfo.getCapabilities().getRequests().setOptionalEnergy(false);
#endif
         msg.addDeviceInfo(devInfo);
         break;
      }

      case MSG_DEVICE_STATUS: {
         SEMP::DeviceStatus devStatus;
         devStatus.setDeviceId(id);
         devStatus.setEMSignalsAccepted(true);
         devStatus.setStatus(_statusOn ? SEMP::StatusRef::On : SEMP::StatusRef::Off);

         /*
          * The device must provide the averaged power of the last minute when it is in On-State.
          */
         if (_statusOn)
         {
            SEMP::PowerInfo powerInfo;
            powerInfo.setAveragePower(500); // average power over last minute
            powerInfo.setTimestamp(0); // must be now (end of measurement in relative time)
            powerInfo.setAveragingInterval(60); // must be one minute
            devStatus.getPowerConsumption().addPowerInfo(powerInfo);
         }

         msg.addDeviceStatus(devStatus);
         break;
      }

      case MSG_TIMEFRAME: {
         // initialize startTime when timeframes are fetched for the first time
         if (_startTime == -1)
            _startTime = Time::getCurrentTime();

         const timestamp_t timeSinceStart = Time::getCurrentTime() - _startTime;

         const timestamp_t relEarliestStart = std::max<timestamp_t>(0, 1000 - timeSinceStart);
         const timestamp_t relLatestEnd = 10000 - timeSinceStart;

         const int runningTime = 7200;

         if (relLatestEnd > 0) {
            SEMP::Timeframe tf;

            tf.setDeviceId(id);
            tf.setEarliestStart(relEarliestStart);
            tf.setLatestEnd(relLatestEnd);
#ifdef SEND_CAN_FRAME
            tf.setMinRunningTime(0);
#else
            // Must-Frame: minRunningTime=maxRunningTime
            tf.setMinRunningTime(runningTime);
#endif
            tf.setMaxRunningTime(runningTime);

            SEMP::PlanningRequest preq;
            preq.addTimeframe(tf);
            msg.addPlanningRequest(preq);
         }

         break;
      }

      default:
         break;

      }
   }

   /*!
    * @brief Called whenever the Home Manager sends a DeviceControl message.
    * A DeviceControl is send, whenever the Home Manager recommends to switch the device on/off.
    *
    * Note: in real environments, the recommendation has to be checked first.
    * If any damage would be caused by following the recommendation, the device has to ignore it.
    * It then should follow the recommendation as soon as possible.
    */
   virtual void controlDevice(const SEMP::DeviceControl &devControl)
   {
      // Follow recommendation.
      _statusOn = devControl.getOn();

      // print contents
      std::stringstream ss;
      ss << "DeviceControl[" << (devControl.getOn() ? "On" : "Off") << "] for deviceId[" << devControl.getDeviceId() << "]";
      Logger::log() << TRED(ss.str()) << std::endl;
   }

private:
   timestamp_t _startTime;
   bool _statusOn;
};


#endif
