/***************************************************************************
 *              SMA Solar Technology AG, 34266 Niestetal, Germany
 ***************************************************************************
 *
 *  Copyright (c) 2008-2014
 *  SMA Solar Technology AG
 *  All rights reserved.
 *  This design is confidential and proprietary of SMA Solar Technology AG.
 *
 ***************************************************************************/

#include "Thread.h"
#include "Logger.h"
#include <sched.h>
#include <stdio.h>
#include <unistd.h>

#ifdef _DEBUG
#include <iostream>
#endif

Thread::Thread() :
   running(false),
   aborted(false),
   threadId(0)
{
   pthread_mutex_init(&dataLock, NULL);
}

void Thread::start() {
   if (isRunning())
      return;

   running = true;
   if (pthread_create(&threadId, NULL, threadFunction, this) != 0)
   {
      running = false;
      return;
   }
   pthread_detach(threadId);
}

Thread::~Thread() {
   if (isRunning())
      abort();
   pthread_mutex_destroy(&dataLock);
}

void Thread::abort(int timeout) {
   const int DELAY_MS = 200;

   setDataLock();
   aborted = true;
   unsetDataLock();

   bool useTimeout = (timeout != -1);
   while (isRunning())
   {
      if (useTimeout)
      {
         if (timeout < 0)
            break;
         timeout -= DELAY_MS;
      }
      usleep(DELAY_MS * 1000);
   }

   if (isRunning()) {
      pthread_cancel(threadId);
   }

   setDataLock();
   running = false;
   unsetDataLock();
}

bool Thread::isAborted() {
   bool tmpAborted;
   setDataLock();
   tmpAborted = aborted;
   unsetDataLock();
   return tmpAborted;
}

bool Thread::isRunning() {
   bool tmpRunning;
   setDataLock();
   tmpRunning = running;
   unsetDataLock();
   return tmpRunning;
}

void Thread::setDataLock() {
   pthread_mutex_lock(&dataLock);
}

void Thread::unsetDataLock() {
   pthread_mutex_unlock(&dataLock);
}

void * Thread::threadFunction(void* arg) {
   int res;
   int oldstate;
   int oldtype;
   Thread * thr = (Thread *) arg;

   res = pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, &oldstate);
   if(res != 0)
      Logger::log() << "Failed to set cancelability state!" << std::endl;
   res = pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, &oldtype);
   if(res != 0)
      Logger::log() << "Failed to set canceltype!" << std::endl;

   thr->run();

   thr->setDataLock();
   thr->running = false;
   thr->unsetDataLock();

   return NULL;
}
