/***************************************************************************
 *              SMA Solar Technology AG, 34266 Niestetal, Germany
 ***************************************************************************
 *
 *  Copyright (c) 2008-2014
 *  SMA Solar Technology AG
 *  All rights reserved.
 *  This design is confidential and proprietary of SMA Solar Technology AG.
 *
 ***************************************************************************/

#ifndef STRINGHELPER_H_
#define STRINGHELPER_H_

#include <string>
#include <vector>

/*! @namespace StringHelper
 *  @brief this namespace comprises functions for string manipulation
 */
namespace StringHelper
{
   /**
    * @brief Splits string @p str separated by the delimiters given in @p delim into tokens.
    *
    * @param str the string to split
    * @param delim a string with each char representing a delimiter
    * @param[out] result a reference to a buffer for the generated tokens
    * @param groupDelimSequence if true, sequences of delimiters are handled as one delimiter (no empty tokens generated for this)
    * @param maxTokenCnt the max. number of tokens to generate. If this number is reached the remaining string is the last token.
    */
   void split(const std::string &str, const std::string &delim, std::vector<std::string> &result,
         bool groupDelimSequence, int maxTokenCnt = -1);

   /**
    * @brief Strips leading whitespace (spaces, tabs, newline) from string @p str
    *
    * @param str the string to strip
    * @return the stripped string
    */
   std::string lstrip(const std::string &str);

   /**
    * @brief Strips trailing whitespace (spaces, tabs, newline) from string @p str
    *
    * @param str the string to strip
    * @return the stripped string
    */
   std::string rstrip(const std::string &str);

   /**
    * @brief Strips leading and trailing whitespace (spaces, tabs, carriage returns, newline) from string @p str
    *
    * @param str the string to strip
    * @return the stripped string
    */
   std::string strip(const std::string &str);

   /**
    * @brief Checks if string @p str starts with string @p start
    *
    * @param str the string to compare
    * @param start the expected start of string
    * @return true if the string starts with the start sequence, false otherwise
    */
   bool startsWith(const std::string &str, const std::string &start);

   /**
    * @brief Checks if string @p str ends  with string @p end
    *
    * @param str the string to compare
    * @param end the expected end of string
    * @return true if the string ends with the end sequence, false otherwise
    */
   bool endsWith(const std::string &str, const std::string &end);

   /**
    * @brief Transforms a given string to all lower case.
    *
    * @param[in] str the string to transform
    */
   std::string toLower(std::string str);

   /**
    * @brief Transforms a given string to all upper case.
    *
    * @param[in] str the string to transform
    */
   std::string toUpper(std::string str);

   /**
    * @brief Extracts a substring from @p str that is surrounded by @p prefix and @p suffix.
    *
    * @param str the string
    * @param prefix the prefix of the substring to extract
    * @param suffix the suffix of the substring to extract
    * @param result the substring found
    * @param pos the search start position in @p str
    * @return the position after the suffix found or std::string::npos if no substring was found
    */
   size_t extract(const std::string &str, const std::string &prefix, const std::string &suffix, std::string &result, size_t pos = 0);

   /**
    * @brief Replaces a substring of @p str with @p replacement.
    *
    * @return @p str
    */
   std::string& replace(std::string &str, const std::string &substr, const std::string &replacement, bool global);

   int compareIgnoreCase(const std::string &str1, const std::string &str2);

   /**
    * @brief Reads the contents of the file given by @p filename into buffer @p buffer.
    *
    * @param filename the filename of the file to read
    * @param[out] a reference to the buffer into which the contents of the file is written to
    * @return true, if reading was successful and the buffer contains the contents of the file
    */
   bool readFromFile(const std::string &filename, std::string &buffer);
}


#endif /* STRINGHELPER_H_ */
