/***************************************************************************
 *              SMA Solar Technology AG, 34266 Niestetal, Germany
 ***************************************************************************
 *
 *  Copyright (c) 2008-2014
 *  SMA Solar Technology AG
 *  All rights reserved.
 *  This design is confidential and proprietary of SMA Solar Technology AG.
 *
 ***************************************************************************/

#ifndef MULTICASTRECEIVER_H_
#define MULTICASTRECEIVER_H_

#include <string>
#include <vector>
#include "Thread.h"
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#define MAX_LEN  4096
#define MIN_PORT 1024
#define MAX_PORT 65535

/*!
 * @brief Encapsulates a network address and netmask.
 */
struct network_iface_t {
   struct in_addr addr;
   in_addr_t netmask;

   network_iface_t(struct in_addr addr_, in_addr_t netmask_) :
      addr(addr_), netmask(netmask_) {}
};

/*! @class IMulticastListener
 *
 * @brief A IMulticastListener is a callback handler called whenever a multicast package was received.
 */
class IMulticastListener {
public:
   virtual void onMulticastReceived(const char *buffer, ssize_t bufSize, const struct sockaddr_in &recAddr) = 0;
   virtual void onInterfaceChanged() = 0;

protected:
   virtual ~IMulticastListener() {}
};

/*! @class MulticastSocket
 *
 * @brief A MulticastSocket listens on a UDP port for multicast packages and calls a notification callback if a packet was received.
 * 
 * It discovers all the network interfaces available on the host and joins the multicast group for them.
 */
class MulticastSocket : protected Thread {
   public:
      
      /** 
      * @brief constructor
      * 
      * @param address address of the multicast group to join
      * @param port port to listen on
      * @param unicastRecv if true, only send multicast datagrams, but receive responses as unicast datagrams,
      *        otherwise send and receive multicast datagrams
      */      
      MulticastSocket(std::string address, int port, bool unicastRecv = false);
      
      /** 
      * @brief destructor
      */      
      virtual ~MulticastSocket();

      /**
      * @brief Starts listening on the multicast interfaces for incoming messages and passes them to the listener
      *
      * @param listener the listener which will receive the incoming messages
      */
      void listen(IMulticastListener *listener);

      /**
      * @brief Returns a list of interfaces this socket is listening to.
      */
      std::vector<network_iface_t> getInterfaces();

      /**
      * @brief Selects the interface used for outgoing messages. If no interface is selected the default routing table entry will be used.
      *
      * @param addr the interfaces address
      */
      void selectInterface(in_addr_t addr);

      /**
      * @brief Sends a UDP packet on the selected interface to the multicast address
      *
      * @param sendBuffer the buffer to send
      * @param length the length of the buffer
      */
      ssize_t send(const char* sendBuffer, size_t length);

      /**
      * @brief Sends a UDP packet to an address that might differ from the multicast address
      *
      * @param sendBuffer the buffer to send
      * @param length the length of the buffer
      * @param dstAddr the destination address
      */
      ssize_t send(const char* sendBuffer, size_t length, const struct sockaddr_in &dstAddr);

   protected:
      bool addMembership();

      bool dropMembership();

      virtual void run();

      /** 
      * @brief Method that allows to discover the interfaces which are available for a socket
      * 
      * @return vector of interfaces
      */      
      std::vector<network_iface_t> enumerateInterfaces();

      /**
      * @brief Update internal interface list according to newly added or removed interfaces
      *
      * @return true if the interfaces changed
      */
      bool updateInterfaces();

      /**
      * @brief Set to true if the socket is listening for incoming multicast packages
      */
      bool _listening;

      /**
      * @brief The listener handling incoming multicast packages
      */
      IMulticastListener *_listener;

      /** 
      * @brief Address of the multicast group to join
      */      
      std::string _address;

      /**
      * @brief The underlying UDP socket
      */
      int _sock;

      /**
      * @brief Interfaces tracked
      */
      std::vector<network_iface_t> _interfaces;

      /** 
      * @brief port to listen on
      */      
      int _port;

      /**
      * @brief if true, only send multicast datagrams, but receive responses as unicast datagrams,
      *        otherwise send and receive multicast datagrams
      */
      bool _unicastRecv;
   
};


#endif
