/***************************************************************************
 *              SMA Solar Technology AG, 34266 Niestetal, Germany
 ***************************************************************************
 *
 *  Copyright (c) 2008-2014
 *  SMA Solar Technology AG
 *  All rights reserved.
 *  This design is confidential and proprietary of SMA Solar Technology AG.
 *
 ***************************************************************************/

#ifndef GATEWAYSIMULATOR_H_
#define GATEWAYSIMULATOR_H_

#include "http/HTTPServer.h"
#include "ssdp/SSDPServer.h"
#include "base/UDPSocket.h"
#include "base/PropertyMap.h"
#include "structs/Device2EM.h"
#include "structs/EM2Device.h"

enum msg_type_t
{
   MSG_SEMP_INFO,
   MSG_DEVICE_INFO,
   MSG_DEVICE_STATUS,
   MSG_TIMEFRAME
};

#define DEFAULT_GATEWAY_HTTP_PORT 8080

/*! @class IGatewayHandler
 *
 * @brief Interface for implementations of gateway behavior.
 *
 * A handler specifies the behavior of the devices managed by a gateway. This includes the SEMP messages
 * sent to the EM like DeviceInfo, DeviceStatus and PlanningRequest but also messages received like
 * power-consumption recommendations in DeviceControl messages.
 */
class IGatewayHandler
{
public:
   /*!
    * @brief Destructor
    */
   virtual ~IGatewayHandler() {}

   /*!
    * @brief Returns a list of device-IDs managed by this handler.
    */
   virtual std::vector<std::string> getDeviceIDs() = 0;

   /*!
    * @brief Handles device control request.
    *
    * @param type the type of message to generate, e.g. MSG_DEVICE_INFO
    * @param msg the message container that will be sent to the EM. Append the requested message to this container.
    * @param id the deviceId the message is associated with (might be an empty string if not applicable, e.g. in case of MSG_SEMP_INFO)
    */
   virtual void getDeviceMsg(msg_type_t type, SEMP::Device2EM &msg, const std::string &id) = 0;

   /*!
    * @brief Handles device control request.
    *
    * @param devControl the device control structure
    */
   virtual void controlDevice(const SEMP::DeviceControl &devControl) = 0;
};

/*! @class GatewaySimulator
 *
 * @brief This class represents a gateway that manages devices controllable
 * with the SimpleEnergyManagementProtocol.
 *
 * It announces itself via SSDP and receives and handles SimpleEnergyManagementProtocol actions.
 */
template <typename GatewayHandlerType>
class GatewaySimulator: public IHTTPRequestHandler
{
public:
   /*!
    * @brief Constructor
    */
   GatewaySimulator(const std::string &uuid, const std::string &serialNumber, uint httpPort = DEFAULT_GATEWAY_HTTP_PORT);

   /*!
    * @brief Destructor
    */
   ~GatewaySimulator();

   /*!
    * @brief Handles a HTTP request
    *
    * @return true if the request could be handled
    */
   virtual bool handleHTTPRequest(const HTTPRequest &request, HTTPResponse &response);

   /*!
    * @brief Returns the BaseURL used by the webserver
    *
    * Format: http://<ip>:<port>
    *
    * @return the BaseURL used by the webserver
    */
   std::string getHttpServerBaseURL() const;

   /*!
    * @brief Starts execution of the gateway.
    */
   void start();

private:
   void setupSSDPServer(const std::string &uuid, const std::string &serialNumber, uint httpPort);

   /*!
    * the SSDP server used to provide information about this gateway and its managed devices
    */
   SSDPServer _ssdpServer;

   /*!
    * the HTTP server used to receive SimpleEnergyManagementProtocol requests
    */
   HTTPServer _httpServer;

   /*!
    * stores deviceIDs of devices managed by this gateway
    */
   std::vector<std::string> devices;

   /*!
    * @brief UPnP root device configuration for this gateway
    */
   SSDPDeviceConfig _deviceConfig;

   GatewayHandlerType handler;
};


#endif
