/***************************************************************************
 *              SMA Solar Technology AG, 34266 Niestetal, Germany
 ***************************************************************************
 *
 *  Copyright (c) 2008-2014
 *  SMA Solar Technology AG
 *  All rights reserved.
 *  This design is confidential and proprietary of SMA Solar Technology AG.
 *
 ***************************************************************************/

#include <sstream>
#include <vector>
#include <iostream>
#include <algorithm>
#include <unistd.h>
#include <signal.h>
#include <stdlib.h>
#include "GatewaySimulator.h"
#include "SEMPCommon.h"
#include "base/Logger.h"
#include "base/Console.h"
#include "base/StringHelper.h"
#include "structs/Device2EM.h"
#include "structs/StructsIndex.h"

#define ENABLE_STORAGE 0

#if ENABLE_STORAGE
#include "handlers/StorageGatewayHandler.h"
typedef StorageGatewayHandler GatewayHandler;
#else
#include "handlers/CommonGatewayHandler.h"
typedef CommonGatewayHandler GatewayHandler;
#endif

#define HTTP_SERVER_VERBOSELEVEL 4

template <typename GatewayHandlerType>
GatewaySimulator<GatewayHandlerType>::GatewaySimulator(const std::string &uuid, const std::string &serialNumber, uint httpPort) :
      _ssdpServer(httpPort),
      _httpServer(this, httpPort)
{
   setupSSDPServer(uuid, serialNumber, httpPort);
   devices = handler.getDeviceIDs();
}

template <typename GatewayHandlerType>
void GatewaySimulator<GatewayHandlerType>::setupSSDPServer(const std::string &uuid, const std::string &serialNumber, uint httpPort) {
   _deviceConfig.uuid = uuid;
   _deviceConfig.urn = SEMP_URN_DEVICE;
   _deviceConfig.locationPath = "/uuid:" + uuid + "/description.xml";
   _deviceConfig.friendlyName = "SimpleEnergyManagementProtocol Gateway";
   _deviceConfig.manufacturer = "SMA Solar Technology AG";
   _deviceConfig.manufacturerURL = "http://www.sma.de";
   _deviceConfig.modelDescription = "SimpleEnergyManagementProtocol Gateway Demo";
   _deviceConfig.modelName = "SEMP GWD";
   _deviceConfig.modelNumber = "1.0.0";
   _deviceConfig.modelURL = "http://www.sma.de";
   _deviceConfig.serialNumber = serialNumber;
   _deviceConfig.presentationURL = "presentation.html";
   // add non-standard baseURL element
   std::stringstream httpPortStr;
   httpPortStr << httpPort;
   _deviceConfig.additionalElements.push_back(
         "<semp:X_SEMPSERVICE xmlns:semp=\"urn:"SEMP_DOMAIN_NAME":service-1-0\">\n"
         "\t<semp:server>http://@IFACE_ADDR@:" + httpPortStr.str() + "</semp:server>\n"
         "\t<semp:basePath>"SEMP_BASE_URL_PATH_DEFAULT"</semp:basePath>\n"
         "\t<semp:transport>HTTP/Pull</semp:transport>\n"
         "\t<semp:exchangeFormat>XML</semp:exchangeFormat>\n"
         "\t<semp:wsVersion>" + SEMP::XSDVersion().str() + "</semp:wsVersion>\n"
         "</semp:X_SEMPSERVICE>\n"
         );
   // set config
   _ssdpServer.setRootDeviceConfig(_deviceConfig);

   // Note: at least one dummy service should be defined,
   // otherwise Windows XP will not display the device in the network environment
   SSDPServiceConfig svc;
   svc.typeUrn = SEMP_DOMAIN_NAME":service:NULL:1";
   svc.idUrn = SEMP_DOMAIN_NAME":serviceId:NULL";
   svc.SCPDURL = "/XD/NULL.xml";
   svc.controlURL = "/UD/?0";
   svc.eventSubURL = "";
   _ssdpServer.addService(svc);
}

template <typename GatewayHandlerType>
GatewaySimulator<GatewayHandlerType>::~GatewaySimulator()
{
   _httpServer.abort();
   _ssdpServer.abort();
}

template <typename GatewayHandlerType>
bool GatewaySimulator<GatewayHandlerType>::handleHTTPRequest(const HTTPRequest &request, HTTPResponse &response)
{
   const std::string baseURI = SEMP_BASE_URL_PATH_DEFAULT;

   // handle UPnP device description request
   if (request.path == "/uuid:" + _deviceConfig.uuid + "/description.xml")
   {
		response.header.fields["Content-Type"] = "text/xml";
      response.body = _ssdpServer.getDeviceDescriptor(request.recvAddr.sin_addr);
      response.header.status = HTTP_STATUS_OK;
      return true;
   }
   else if (request.path == "/uuid:" + _deviceConfig.uuid + "/presentation.html")
   {
		response.header.fields["Content-Type"] = "text/html";
      response.body = "<html>\n"
            "\t<head><title>SimpleEnergyManagementProtocol</title></head>\n"
            "\t<body><p>SEMP Gateway Demo<p></body>\n"
            "</html>\n";
      response.header.status = HTTP_STATUS_OK;
      return true;
   }
   // handle SimpleEnergyManagementProtocol device action
   else if (StringHelper::startsWith(request.path, baseURI))
   {
      // remove baseURI and remove trailing '/' if existing
      std::string cmd = request.path.substr(baseURI.size());
      if (cmd[cmd.size() - 1] == '/')
         cmd.erase(cmd.size() - 1);

      std::string deviceId;
      propertyMap_t::const_iterator queryDevIt = request.query.find("DeviceId");
      if (queryDevIt != request.query.end())
         deviceId = queryDevIt->second;

      if (request.method == HTTP_METHOD_POST)
      {
         // DeviceControl or other data Submission via POST: extract new state and deviceId from body

         size_t pos = 0;
         std::string devControlBuffer;
         while ((pos = StringHelper::extract(request.body, "<DeviceControl", "</DeviceControl>", devControlBuffer, pos)) != std::string::npos)
         {
            SEMP::DeviceControl devControl;

            if (StringHelper::extract(devControlBuffer, "<DeviceId>", "</DeviceId>", devControl.getDeviceId()) == std::string::npos)
            {
               Logger::log() << "Element <DeviceId> not found in DeviceControl request" << std::endl;
               continue;
            }

            if (std::find(devices.begin(), devices.end(), devControl.getDeviceId()) == devices.end())
            {
               Logger::log() << "Ignoring control-message for deviceId '" << devControl.getDeviceId() + "', not managed by gateway" << std::endl;
               continue;
            }

            std::string timestampStr;
            if (StringHelper::extract(devControlBuffer, "<Timestamp>", "</Timestamp>", timestampStr) != std::string::npos)
            {
               devControl.setTimestamp(atoll(timestampStr.c_str()));
            }

            std::string onStr;
            if (StringHelper::extract(devControlBuffer, "<On>", "</On>", onStr) != std::string::npos)
            {
               devControl.setOn(onStr == "true");
            }

            std::string powerStr;
            if (StringHelper::extract(devControlBuffer, "<RecommendedPowerConsumption>", "</RecommendedPowerConsumption>", powerStr) != std::string::npos)
            {
               devControl.setRecommendedPowerConsumption(atof(powerStr.c_str()));
            }

#if ENABLE_STORAGE
            std::string storageStr;
            if (StringHelper::extract(devControlBuffer, "<Storage>", "</Storage>", storageStr) != std::string::npos)
            {
               std::string upperLimitStr;
               if (StringHelper::extract(storageStr, "<UpperPLim>", "</UpperPLim>", upperLimitStr) != std::string::npos)
               {
                  devControl.getStorage().setUpperPLim(atof(upperLimitStr.c_str()));
               }

               std::string lowerLimitStr;
               if (StringHelper::extract(storageStr, "<LowerPLim>", "</LowerPLim>", lowerLimitStr) != std::string::npos)
               {
                  devControl.getStorage().setLowerPLim(atof(lowerLimitStr.c_str()));
               }
            }
#endif

            handler.controlDevice(devControl);
         }

         response.header.status = HTTP_STATUS_OK;
         return true;
      }
      else if (request.method == HTTP_METHOD_GET)
      {
         response.header.fields["Content-Type"] = "application/xml";

         SEMP::Device2EM msg;

         // "baseURI/": return all information available (all DeviceInfo, DeviceStatus, PlanningRequest structures)
         if (cmd == "")
         {
            handler.getDeviceMsg(MSG_SEMP_INFO, msg, "");
            for (size_t i = 0; i < devices.size(); ++i)
               handler.getDeviceMsg(MSG_DEVICE_INFO, msg, devices[i]);
            for (size_t i = 0; i < devices.size(); ++i)
               handler.getDeviceMsg(MSG_DEVICE_STATUS, msg, devices[i]);
            for (size_t i = 0; i < devices.size(); ++i)
               handler.getDeviceMsg(MSG_TIMEFRAME, msg, devices[i]);
         }
         // "baseURI/SEMPInfo": return SEMPInfo data-structures
         else if (cmd == "/SEMPInfo")
         {
            handler.getDeviceMsg(MSG_DEVICE_INFO, msg, "");
         }
         // "baseURI/DeviceInfo": return DeviceInfo data-structures
         else if (cmd == "/DeviceInfo")
         {
            for (size_t i = 0; i < devices.size(); ++i)
            {
               if (deviceId == "" || deviceId == devices[i])
                  handler.getDeviceMsg(MSG_DEVICE_INFO, msg, devices[i]);
            }
         }
         // "baseURI/DeviceStatus": return DeviceStatus data-structures
         else if (cmd == "/DeviceStatus")
         {
            for (size_t i = 0; i < devices.size(); ++i)
            {
               if (deviceId == "" || deviceId == devices[i])
                  handler.getDeviceMsg(MSG_DEVICE_STATUS, msg, devices[i]);
            }
         }
         // "baseURI/PlanningRequest": return PlanningRequest data-structures
         else if (cmd == "/PlanningRequest")
         {
            for (size_t i = 0; i < devices.size(); ++i)
            {
               if (deviceId == "" || deviceId == devices[i])
                  handler.getDeviceMsg(MSG_TIMEFRAME, msg, devices[i]);
            }
         }
         else
         {
            Logger::log() << "Unknown device action" << std::endl;
            return false;
         }

         std::string buffer;
         if (!msg.validateXMLSchema(buffer)) {
            Logger::log() << "XML validation failed: " << buffer << std::endl;
            return false;
         }

         std::stringstream ss;
         msg.serializeXML(ss);
         response.body = ss.str();
         response.header.status = HTTP_STATUS_OK;
         return true;
      }
   }

   Logger::log() << "Unknown action" << std::endl;
   return false;
}

template <typename GatewayHandlerType>
std::string GatewaySimulator<GatewayHandlerType>::getHttpServerBaseURL() const {
   return _httpServer.getBaseURL();
}

template <typename GatewayHandlerType>
void GatewaySimulator<GatewayHandlerType>::start() {
   Logger::log() << "Using SEMP XSD version: " << SEMP::XSDVersion().str() << std::endl;

   Logger::log() << "Starting HTTP-Server ..." << std::endl;
   _httpServer.setVerboseLevel(HTTP_SERVER_VERBOSELEVEL);
   _httpServer.start();

   Logger::log() << "Starting SSDP-Server ..." << std::endl;
   //_ssdpServer.setNotifyInterval(10);
   _ssdpServer.start();

   Logger::log() << "Gateway ready" << std::endl;
}



// gateway instance stored for signal handler
static GatewaySimulator<GatewayHandler> *gateway;

/*
 * Signal handler for CTRL-C and CTRL+\
 */
void signal_handler(int signum)
{
   static bool aborting = false;
   if (!aborting) {
      aborting = true;
      std::cout << "Shutting-down Gateway ..." << std::endl;
      delete gateway;
      exit(0);
   }
}

/*
 * Demo gateway configuration
 */
#define GATEWAY_SERIAL  "53-4D-41-53-4D-41"
// UPnP v1.0 does not specify a UUID format, the definition from UPnP v1.1 is used here
#define GATEWAY_UUID    "2fac1234-31f8-11b4-a222-08002b34c003"

/*
 * Main routine for EnergyManager demo
 */
int main() {
   signal(SIGQUIT, signal_handler);
   signal(SIGINT, signal_handler);

   gateway = new GatewaySimulator<GatewayHandler>(GATEWAY_UUID, GATEWAY_SERIAL);
   gateway->start();

   while (true)
      pause();
}
