/***************************************************************************
 *              SMA Solar Technology AG, 34266 Niestetal, Germany
 ***************************************************************************
 *
 *  Copyright (c) 2008-2014
 *  SMA Solar Technology AG
 *  All rights reserved.
 *  This design is confidential and proprietary of SMA Solar Technology AG.
 *
 ***************************************************************************/

#include <iostream>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <set>
#include <sstream>
#include "base/MulticastSocket.h"
#include "base/StringHelper.h"
#include "base/Console.h"
#include "base/Logger.h"
#include "structs/EM2Device.h"
#include "structs/StructsIndex.h"
#include "SEMPCommon.h"
#include "EnergyManager.h"

EnergyManager::EnergyManager() :
   _verbosity(0),
   _interactive(true),
   _ssdpDiscovery(this)
{}

EnergyManager::~EnergyManager()
{
   _ssdpDiscovery.abort();
}

void EnergyManager::waitForKeypress() {
   if (_interactive) {
      Logger::log() << "<Press Enter to continue>" << std::endl << std::endl;
      waitNewline();
   }
}

void EnergyManager::updateSSDPDevice(SSDPDeviceState state, const SSDPDeviceInfo &info)
{
   if (state == STATE_BYEBYE)
      return;

   if (_knownGateways.find(info.uuid) != _knownGateways.end())
      return;
   _knownGateways.insert(info.uuid);

   Logger::log() << TRED("EnergyManager: found a new Gateway:\n  UUID: " << info.uuid) << std::endl;
   waitForKeypress();

   Logger::log() << std::endl << "EnergyManager: request UPnP descriptor from Gateway (" << info.location << ")" << std::endl;
   waitForKeypress();

   // get gateway description

   std::string server = "http://"
         + std::string(inet_ntoa(info.devAddr.sin_addr))
         + ":" + STR(SEMP_BASE_URL_PORT_DEFAULT);
   std::string basePath = SEMP_BASE_URL_PATH_DEFAULT;

   if (_httpClient.performRequest(HTTP_METHOD_GET, info.location) != 0)
   {
      Logger::log() << "EnergyManager: failed to retrieve UPnP device descriptor!" << std::endl;
      _knownGateways.erase(info.uuid);
      return;
   }

   std::string response;

   response = _httpClient.getResponseBody();
   Logger::log() << "EnergyManager: received UPnP device descriptor from Gateway:\n" << TCYAN(response) << std::endl << std::endl;

   // extract SimpleEnergyManagementProtocol baseURL
   std::string sempNamespace;
   // check if namespace for SEMP service is present (note: namespace declaration might be inside SEMP service element)
   if (StringHelper::extract(response, "xmlns:", "=\"urn:"SEMP_DOMAIN_NAME":service-1-0\"", sempNamespace) != std::string::npos)
   {
      size_t elPos;

      // extract server address
      elPos = response.find("<"+sempNamespace+":server");
      if (elPos != std::string::npos)
      {
         if (StringHelper::extract(response, ">", "</"+sempNamespace+":server>", server, elPos) != std::string::npos)
            Logger::log() << "EnergyManager: found SEMP server information '" << server << "'" << std::endl;
      }

      // extract baseURL
      elPos = response.find("<"+sempNamespace+":basePath");
      if (elPos != std::string::npos)
      {
         if (StringHelper::extract(response, ">", "</"+sempNamespace+":basePath>", basePath, elPos) != std::string::npos)
            Logger::log() << "EnergyManager: found SEMP basePath information '" << basePath << "'" << std::endl;
      }
   }

   // fallback to default URL
   std::string baseURL = server + basePath;
   Logger::log() << "EnergyManager: baseURL '" + baseURL + "'" << std::endl;

   // request all available information for all devices from Gateway

   Logger::log() << std::endl << "EnergyManager: request all available information from Gateway" << std::endl;
   waitForKeypress();

   std::vector<std::string> devices;

   if (_httpClient.performRequest(HTTP_METHOD_GET, baseURL + "/") != 0)
   {
      Logger::log() << "EnergyManager: failed to retrieve information from Gateway!" << std::endl;
      return;
   }

   if (_httpClient.getResponseHeader().getStatusCode() != HTTP_STATUS_OK) {
      Logger::warn() << "EnergyManager: invalid response (HTTP status code: " << _httpClient.getResponseHeader().getStatusCode() << ")" << std::endl;
      return;
   }

   response = _httpClient.getResponseBody();
   Logger::log() << "EnergyManager: received information from Gateway\n" << TRED(response) << std::endl << std::endl;

   // extract device information
   size_t pos = 0;
   std::string devInfo;
   while ((pos = StringHelper::extract(response, "<DeviceInfo", "</DeviceInfo>", devInfo, pos)) != std::string::npos)
   {
      std::string deviceId;
      if (StringHelper::extract(devInfo, "<DeviceId>", "</DeviceId>", deviceId) == std::string::npos)
      {
         Logger::log() << "EnergyManager: no 'DeviceId' element in 'DeviceInfo' found" << std::endl;
         continue;
      }

      devices.push_back(deviceId);
   }

   if (devices.size() == 0)
   {
      Logger::log() << "EnergyManager: Gateway does not manage any device!" << std::endl;
      return;
   }

   Logger::log() << "EnergyManager: Gateway manages " << devices.size() << " devices:" << std::endl;
   for (size_t i = 0; i < devices.size(); i++)
      Logger::log() << "  " << (i+1) << ": " << devices[i] << std::endl;
   Logger::log() << std::endl;

   // request device info for first device from Gateway

   Logger::log() << std::endl << "EnergyManager: request device info of device '" << devices[0] << "' from Gateway" << std::endl;
   waitForKeypress();

   if (_httpClient.performRequest(HTTP_METHOD_GET, baseURL + "/DeviceInfo?DeviceId=" + devices[0]) == 0)
   {
      if (_httpClient.getResponseHeader().getStatusCode() != HTTP_STATUS_OK) {
         Logger::warn() << "EnergyManager: invalid response (HTTP status code: " << _httpClient.getResponseHeader().getStatusCode() << ")" << std::endl;
      } else {
         Logger::log() << "EnergyManager: received DeviceInfo for device '" << devices[0] << "' from Gateway:" << std::endl
               << TGREEN(_httpClient.getResponseBody()) << std::endl;
      }
   }

   // request device status for first device from Gateway

   Logger::log() << std::endl << "EnergyManager: request device status of device '" << devices[0] << "' from Gateway" << std::endl;
   waitForKeypress();

   if (_httpClient.performRequest(HTTP_METHOD_GET, baseURL + "/DeviceStatus?DeviceId=" + devices[0]) == 0)
   {
      if (_httpClient.getResponseHeader().getStatusCode() != HTTP_STATUS_OK) {
         Logger::warn() << "EnergyManager: invalid response (HTTP status code: " << _httpClient.getResponseHeader().getStatusCode() << ")" << std::endl;
      } else {
         Logger::log() << "EnergyManager: received DeviceStatus for device '" << devices[0] << "' from Gateway:" << std::endl
               << TBLUE(_httpClient.getResponseBody()) << std::endl;
      }
   }

   // request timeframe for first device from Gateway

   Logger::log() << std::endl << "EnergyManager: perform a planning request of device '" << devices[0] << "' from Gateway" << std::endl;
   waitForKeypress();

   if (_httpClient.performRequest(HTTP_METHOD_GET, baseURL + "/PlanningRequest?DeviceId=" + devices[0]) == 0)
   {
      if (_httpClient.getResponseHeader().getStatusCode() != HTTP_STATUS_OK) {
         Logger::warn() << "EnergyManager: invalid response (HTTP status code: " << _httpClient.getResponseHeader().getStatusCode() << ")" << std::endl;
      } else {
         Logger::log() << "EnergyManager: received PlanningRequest for device '" << devices[0] << "' from Gateway:" << std::endl
               << TYELLOW(_httpClient.getResponseBody()) << std::endl;
      }
   }

   // send control message to device (via POST)

   for (size_t i = 0; i < devices.size(); ++i)
   {
      Logger::log() << std::endl << "EnergyManager: send control message to device '" << devices[i] << "' (with HTTP-POST method) via Gateway" << std::endl;
      waitForKeypress();

      SEMP::EM2Device msg;
      SEMP::DeviceControl devControl;
      devControl.setDeviceId(devices[i]);
      devControl.setOn(true);
      // Note: min/maxPowerConsumption of the device is ignored in the example
      //devControl.setRecommendedPowerConsumption(1000 + i*500);
      devControl.setTimestamp(time(NULL));
      msg.addDeviceControl(devControl);
      sendDeviceRequest(baseURL + "/", msg);
   }

   Logger::log() << std::endl << "EnergyManager: waiting for next Gateway ..." << std::endl;
}

bool EnergyManager::sendDeviceRequest(const std::string &url, const SEMP::EM2Device &msg)
{
   std::string buffer;
   if (!msg.validateXMLSchema(buffer)) {
      Logger::log() << "XML validation failed: " << buffer << std::endl;
      return false;
   } else {
      // for POST DeviceControl use baseURL
      HTTPRequestHeader httpReqHdr(HTTP_METHOD_POST);
      httpReqHdr.fields["Content-Type"] = "application/xml";

      std::stringstream ss;
      msg.serializeXML(ss);
      if (_httpClient.performRequest(httpReqHdr, url, ss.str()) != 0)
      {
         Logger::log() << "Could not send DeviceControl" << std::endl;
         return false;
      }

      if (_httpClient.getResponseHeader().getStatusCode() != HTTP_STATUS_OK) {
         Logger::warn() << "Could not send DeviceControl (HTTP status code: " << _httpClient.getResponseHeader().getStatusCode() << ")" << std::endl;
         return false;
      }

      return true;
   }
}

void EnergyManager::setVerbosity(int level)
{
   _verbosity = level;
}

void EnergyManager::setInteractive(bool interactive)
{
   _interactive = interactive;
}

void EnergyManager::start()
{
   Logger::log() << "Using SEMP XSD version: " << SEMP::XSDVersion().str() << std::endl;
   Logger::log() << "EnergyManager started, searching for Gateways ...\n" << std::endl;

   _ssdpDiscovery.setVerbosity(_verbosity);
   _ssdpDiscovery.addSearchTarget("urn:"SEMP_URN_DEVICE);
   _ssdpDiscovery.start();
   while (true)
      pause();
}


void printUsage(char* progName) {
   std::cout << "Usage: " << progName << " [-v] [-n]" << std::endl
         << "Parameters:" << std::endl
         << "  -v: verbose" << std::endl
         << "  -n: non-interactive (do not wait for key-presses)" << std::endl
         << std::endl;
}

/*
 * Main routine for EnergyManager demo
 */
int main(int argc, char* args[]) {
   EnergyManager energyManager;

   for (int i = 1; i < argc; ++i)
   {
      if (strcmp(args[i], "-v") == 0)
      {
         energyManager.setVerbosity(2);
      }
      else if (strcmp(args[i], "-n") == 0)
      {
         energyManager.setInteractive(false);
      }
      else
      {
         printUsage(args[0]);
         return 0;
      }
   }

   energyManager.start();
}
