#ifndef CAPABILITIES_H_
#define CAPABILITIES_H_

#include "../SEMPCommon.h"
#include "CapPowerMeasurement.h"
#include "CapTimestamp.h"
#include "CapInterruptions.h"
#include "CapRequests.h"

namespace SEMP
{

/*! \class Capabilities
 * Encapsulates information about the capabilities of the device
 */
class Capabilities : public SEMPObject
{
public:
   Capabilities();
   Capabilities(const Capabilities& rhs);
   Capabilities& operator=(const Capabilities& rhs);
private:
   /*!
    * Capability of the device with regard to deriving information about its current power consumption, e.g. measurement or estimation
    */
   CapPowerMeasurement currentPower;

   /*!
    * Specifies if the device is able to deal with absolute timestamps or only with relative timestamps.
            Devices that do not have a synchronized clock (with time server protocols like NTP or radio control like DCF77) or do not have a reliable absolute time source should use relative timestamps.
    */
   CapTimestamp timestamps; /* occurs: 0 .. 1 */

   /*!
    * Specifies if a run of the device can be interrupted or not.
    */
   CapInterruptions interruptions; /* occurs: 0 .. 1 */

   /*!
    * Specifies options related to planning requests.
    */
   CapRequests requests; /* occurs: 0 .. 1 */


public:
   /*!
    * @brief Setter for CapPowerMeasurement
    * Capability of the device with regard to deriving information about its current power consumption, e.g. measurement or estimation
    */
   void setCurrentPower(CapPowerMeasurement const &value_) {
      currentPower = value_;
      currentPower.setOnSetListener(this, "");
      markAsSet("currentPower");
   }

   /*!
    * @brief Getter for CapPowerMeasurement
    * Capability of the device with regard to deriving information about its current power consumption, e.g. measurement or estimation
    */
   CapPowerMeasurement& getCurrentPower() {
      return currentPower;
   }

   /*!
    * @brief Getter for CapPowerMeasurement
    * Capability of the device with regard to deriving information about its current power consumption, e.g. measurement or estimation
    */
   const CapPowerMeasurement& getCurrentPower() const {
      return currentPower;
   }

   /*!
    * @brief Checks if field CapPowerMeasurement is set
    */
   bool isCurrentPowerSet() const {
      return isSet("currentPower");
   }

   /*!
    * @brief Setter for CapTimestamp
    * Specifies if the device is able to deal with absolute timestamps or only with relative timestamps.
            Devices that do not have a synchronized clock (with time server protocols like NTP or radio control like DCF77) or do not have a reliable absolute time source should use relative timestamps.
    */
   void setTimestamps(CapTimestamp const &value_) {
      timestamps = value_;
      timestamps.setOnSetListener(this, "");
      markAsSet("timestamps");
   }

   /*!
    * @brief Getter for CapTimestamp
    * Specifies if the device is able to deal with absolute timestamps or only with relative timestamps.
            Devices that do not have a synchronized clock (with time server protocols like NTP or radio control like DCF77) or do not have a reliable absolute time source should use relative timestamps.
    */
   CapTimestamp& getTimestamps() {
      return timestamps;
   }

   /*!
    * @brief Getter for CapTimestamp
    * Specifies if the device is able to deal with absolute timestamps or only with relative timestamps.
            Devices that do not have a synchronized clock (with time server protocols like NTP or radio control like DCF77) or do not have a reliable absolute time source should use relative timestamps.
    */
   const CapTimestamp& getTimestamps() const {
      return timestamps;
   }

   /*!
    * @brief Checks if field CapTimestamp is set
    */
   bool isTimestampsSet() const {
      return isSet("timestamps");
   }

   /*!
    * @brief Setter for CapInterruptions
    * Specifies if a run of the device can be interrupted or not.
    */
   void setInterruptions(CapInterruptions const &value_) {
      interruptions = value_;
      interruptions.setOnSetListener(this, "");
      markAsSet("interruptions");
   }

   /*!
    * @brief Getter for CapInterruptions
    * Specifies if a run of the device can be interrupted or not.
    */
   CapInterruptions& getInterruptions() {
      return interruptions;
   }

   /*!
    * @brief Getter for CapInterruptions
    * Specifies if a run of the device can be interrupted or not.
    */
   const CapInterruptions& getInterruptions() const {
      return interruptions;
   }

   /*!
    * @brief Checks if field CapInterruptions is set
    */
   bool isInterruptionsSet() const {
      return isSet("interruptions");
   }

   /*!
    * @brief Setter for CapRequests
    * Specifies options related to planning requests.
    */
   void setRequests(CapRequests const &value_) {
      requests = value_;
      requests.setOnSetListener(this, "");
      markAsSet("requests");
   }

   /*!
    * @brief Getter for CapRequests
    * Specifies options related to planning requests.
    */
   CapRequests& getRequests() {
      return requests;
   }

   /*!
    * @brief Getter for CapRequests
    * Specifies options related to planning requests.
    */
   const CapRequests& getRequests() const {
      return requests;
   }

   /*!
    * @brief Checks if field CapRequests is set
    */
   bool isRequestsSet() const {
      return isSet("requests");
   }

   virtual void serializeXML(std::stringstream &sstream, int level_ = 0) const;
   virtual bool validateXMLSchema(std::string &errElement_) const;
};

}

#endif
