/***************************************************************************
 *              SMA Solar Technology AG, 34266 Niestetal, Germany
 ***************************************************************************
 *
 *  Copyright (c) 2008-2014
 *  SMA Solar Technology AG
 *  All rights reserved.
 *  This design is confidential and proprietary of SMA Solar Technology AG.
 *
 ***************************************************************************/

#ifndef SSDPSERVER_H_
#define SSDPSERVER_H_

#include "base/Thread.h"
#include "base/MulticastSocket.h"
#include "base/UDPSocket.h"
#include "SSDPCommon.h"

enum SSDPDiscoveryType {
   SSDP_DT_ROOTDEVICE,
   SSDP_DT_DEVICE_TYPE,
   SSDP_DT_DEVICEID_TYPE,
   SSDP_DT_SERVICE_TYPE,
};

struct SSDPDeviceConfig
{
   std::string uuid;
   std::string urn; // device-type URN. Format: "[schemas-upnp-org|<domain-name>]:device:<deviceType>:<v>" (no "urn:"-prefix)
   std::string locationPath;
   std::string friendlyName;
   std::string manufacturer;
   std::string manufacturerURL;
   std::string modelDescription;
   std::string modelName;
   std::string modelNumber;
   std::string modelURL;
   std::string serialNumber;
   std::string presentationURL;
   std::vector<std::string> additionalElements;
};

struct SSDPServiceConfig
{
   std::string typeUrn;       // service-type URN. Format: "[schemas-upnp-org|<domain-name>]:service:<serviceType>:<v>" (no "urn:"-prefix)
   std::string idUrn;         // service-Id URN. Format: "[upnp-org|<domain-name>]:serviceId:<serviceID>" (no "urn:"-prefix)
   std::string SCPDURL;       // URL to service descriptor
   std::string controlURL;    // URL for control
   std::string eventSubURL;   // URL for eventing
};

/*! @class SSDPServer
 *
 * @brief Provides information using the SSDP protocol to clients implementing SSDP discovery.
 */
class SSDPServer : public Thread, private IMulticastListener
{
public:
   /*!
    * @brief Constructor.
    *
    * @param httpPort the port the webserver listens too
    */
   SSDPServer(int httpPort);

   /*!
    * @brief Destructor.
    */
   ~SSDPServer();

   void setNotifyInterval(int notifyInterval);

   /*!
    * @brief Configures the root device managed by this server.
    *
    * Must be called before the server is started.
    */
   void setRootDeviceConfig(const SSDPDeviceConfig &rootDeviceConfig);

   /*!
    * @brief Returns the UPnP device descriptor of the device handled by this SSDP server.
    */
   std::string getDeviceDescriptor(const struct in_addr &addr) const;

   void addService(const SSDPServiceConfig &svc);

   /*!
    * @brief Thread main routine.
    * @see Thread::run()
    */
   virtual void run();

private:
   /*!
    * @brief multicast socket used by this server
    */
   MulticastSocket *_mcSock;

   /*!
    * @brief UPnP device configuration
    */
   SSDPDeviceConfig _rootDevice;

   /*!
    * @brief UPnP service list
    */
   std::vector<SSDPServiceConfig> _services;

   /*!
    * @brief port of the associated webserver
    */
   int _httpPort;

   /*!
    * @brief regular interval between two notify messages
    */
   int _notifyInterval;

   /*!
    * @brief if true, forces immediate sending of notify messages.
    *
    * This resets the regular wait interval, so that the next message after the forced one
    * will be sent after the regular interval.
    */
   bool _forceSendNotify;

   /*!
    * @brief returns the NT (or ST) and USN field information
    *
    * @param type the SSDP discovery request type
    * @param useSearchTarget if true return ST instead of NT field
    */
   std::string getDiscoveryNT_USN(SSDPDiscoveryType type, bool useSearchTarget) const;

   /*!
    * @brief returns the NT (or ST) and USN field information for UPnP service types
    *
    * @param svc the UPnP service
    * @param useSearchTarget if true return ST instead of NT field
    */
   std::string getDiscoveryServiceNT_USN(const SSDPServiceConfig &svc, bool useSearchTarget) const;

   /*!
    * @brief Multicasts an SSDP Notify message
    *
    * @param alive if true sends an ssdp:alive otherwise ssdp:byebye message
    */
   void sendNotify(bool alive);

   /*!
    * @brief Multicasts an SSDP M-Search response message
    *
    * @param searchTarget the search target specified by the M-Search request
    * @param ifaceAddr the interface address used to handle incoming requests
    * @param dstAddr the destination address of the response
    */
   void sendMSearchResponse(const std::string &searchTarget, const struct in_addr &ifaceAddr, const struct sockaddr_in &dstAddr);

   bool findLocalInterface(const struct in_addr &remoteAddr, struct in_addr &localAddr) const;

   /*!
    * @brief called when the underlying multicast socket receives a new packet
    */
   virtual void onMulticastReceived(const char *buffer, ssize_t bufSize, const struct sockaddr_in &recAddr);

   virtual void onInterfaceChanged();
};

#endif
