/***************************************************************************
 *              SMA Solar Technology AG, 34266 Niestetal, Germany
 ***************************************************************************
 *
 *  Copyright (c) 2008-2014
 *  SMA Solar Technology AG
 *  All rights reserved.
 *  This design is confidential and proprietary of SMA Solar Technology AG.
 *
 ***************************************************************************/

#ifndef HTTPSERVER_H_
#define HTTPSERVER_H_

#include <map>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "base/Thread.h"
#include "base/PropertyMap.h"
#include "HTTPCommon.h"

class HTTPRequestHeaderBuffer : public HTTPMessageHeaderBuffer
{
public:
   HTTPMethod getMethod();
   std::string getURL();

   static void splitQuery(const std::string &url, std::string &path, propertyMap_t &query);
};

/*!
 * @brief Encapsulates the data of an HTTP request.
 */
struct HTTPRequest
{
	HTTPMethod method;
	struct sockaddr_in recvAddr;
	std::string path;
	propertyMap_t query;
	HTTPRequestHeaderBuffer header;
	std::string body;
};

struct HTTPResponseHeader
{
   HTTPStatus status;
   propertyMap_t fields;
};

/*!
 * @brief Encapsulates the data of an HTTP response.
 */
struct HTTPResponse
{
   HTTPResponseHeader header;
	std::string body;
};

/*! @class IHTTPRequestHandler
 *
 * @brief This interface defines a callback to handle incoming HTTP requests.
 */
class IHTTPRequestHandler
{
public:
   /*!
    * Called to handle incoming requests.
    */
   virtual bool handleHTTPRequest(const HTTPRequest &request, HTTPResponse &response) = 0;

protected:
   virtual ~IHTTPRequestHandler() {}
};

/*! @class HTTPServer
 *
 * @brief This class implements a simple HTTP server.
 */
class HTTPServer : public Thread
{
public:
   /*!
    * @brief Constructor
    */
   HTTPServer(IHTTPRequestHandler *handler_, uint port_);

   /*!
    * @brief Returns the BaseURL used by the server
    *
    * Format: http://<ip>:<port>
    */
   std::string getBaseURL() const;

   void setVerboseLevel(int level) {
      _verboseLevel = level;
   }

   /*!
    * @brief Thread main routine.
    * @see Thread::run()
    */
   virtual void run();

private:
   /*!
    * @brief the handler called to handle incoming requests.
    */
   IHTTPRequestHandler *_handler;

   /*!
    * @brief the port listened to.
    */
   uint _port;

   int _verboseLevel;

   /*!
    * @brief Called for each incoming client HTTP requests.
    *
    * @param recvAddr the address and port of the client
    * @param the socket that received the request
    */
   void handleClient(const sockaddr_in &recvAddr, int sock);
};

#endif
