/***************************************************************************
 *              SMA Solar Technology AG, 34266 Niestetal, Germany
 ***************************************************************************
 *
 *  Copyright (c) 2008-2014
 *  SMA Solar Technology AG
 *  All rights reserved.
 *  This design is confidential and proprietary of SMA Solar Technology AG.
 *
 ***************************************************************************/

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <pthread.h>
#include <map>

#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <sstream>
#include <fstream>
#include <iostream>
#include <string>
#include <vector>

#include "HTTPServer.h"
#include "HTTPCommon.h"
#include "base/Logger.h"
#include "base/Console.h"
#include "base/StringHelper.h"

#undef _STR
#undef STR
#define _STR(a) #a
#define STR(a) _STR(a)

#define MAX_METHOD_LENGTH  16
// the HTTP standard does not define a url length limit
#define MAX_URL_LENGTH     1024

HTTPMethod HTTPRequestHeaderBuffer::getMethod()
{
   char method[MAX_METHOD_LENGTH+1];
   if (sscanf(buffer.c_str(), "%"STR(MAX_METHOD_LENGTH)"s %*s HTTP/%*d.%*d", method) != 1)
      return HTTP_METHOD_UNKNOWN;

   if (strcmp(method, "GET") == 0)
      return HTTP_METHOD_GET;
   else if (strcmp(method, "POST") == 0)
      return HTTP_METHOD_POST;
   else
      return HTTP_METHOD_UNKNOWN;
}

std::string HTTPRequestHeaderBuffer::getURL()
{
   char url[MAX_URL_LENGTH+1];
   if (sscanf(buffer.c_str(), "%*s %"STR(MAX_URL_LENGTH)"s HTTP/%*d.%*d", url) != 1)
      return "";
   return std::string(url);
}

void HTTPRequestHeaderBuffer::splitQuery(const std::string &url, std::string &path, propertyMap_t &query)
{
   query.clear();

   std::vector<std::string> tokens;
   StringHelper::split(url, "?", tokens, false);
   path = tokens[0];
   if (tokens.size() == 1) {
      return;
   }

   std::vector<std::string> queryTokens;
   StringHelper::split(tokens[1], "&", queryTokens, false);
   for (std::vector<std::string>::iterator queryIt = queryTokens.begin(); queryIt != queryTokens.end(); queryIt++) {
      std::vector<std::string> keyValue;
      StringHelper::split(*queryIt, "=", keyValue, false);
      if (keyValue.size() < 2)
         query[keyValue[0]] = "";
      else
         query[keyValue[0]] = keyValue[1];
   }
}

HTTPServer::HTTPServer(IHTTPRequestHandler *handler_, uint port_) :
   _handler(handler_),
   _port(port_),
   _verboseLevel(0)
{}

std::string HTTPServer::getBaseURL() const
{
   std::stringstream ss;
   ss << "http://127.0.0.1:" << _port;
   return ss.str();
}

void printRequest(const HTTPRequest &request, int verboseLevel) {
   if (verboseLevel == 0)
      return;

   if (verboseLevel == 1) {
      Logger::log() << "HTTPServer: received HTTP Request: " << request.path << std::endl;
   } else {
      Logger::log(false)
        << "HTTPServer: received HTTP Request:\n"
        << "=====================================================" << std::endl
        << TGREEN(request.header.buffer);
   }
   if (verboseLevel > 2) {
      if (!request.body.empty()) {
         Logger::log(false)
              << "_____________________________________________________" << std::endl
              << TBLUE(request.body);
      }
   }
   Logger::log(false)
        << "=====================================================" << std::endl
        << std::endl;
}

void HTTPServer::handleClient(const sockaddr_in &recvAddr, int sock)
{
   HTTPRequest request;
   request.recvAddr = recvAddr;

   HTTPResponse response;
   response.header.status = HTTP_STATUS_NOT_FOUND;

   if (_verboseLevel > 0)
   {
      Logger::log() << "[Client from " << inet_ntoa(recvAddr.sin_addr) << "]" << std::endl;
   }

   if (HTTPCommon::recvMessage(sock, false, request.header.buffer, request.body, 10))
   {
      printRequest(request, _verboseLevel);

      request.method = request.header.getMethod();
      if (_handler && request.method != HTTP_METHOD_UNKNOWN)
      {
         HTTPRequestHeaderBuffer::splitQuery(request.header.getURL(), request.path, request.query);
         _handler->handleHTTPRequest(request, response);
      }
   }

   std::stringstream responseStream;
   // add header status line and basic fields
   responseStream
         << "HTTP/1.1 " << response.header.status << " " << HttpStatusMsg(response.header.status) << "\r\n"
         << "Content-Length: " << response.body.size() << "\r\n"
         << "Connection: close\r\n";
   // add additional fields specified by the handler
   for (propertyMap_t::iterator it = response.header.fields.begin();
         it != response.header.fields.end(); ++it)
   {
      responseStream << it->first << ": " << it->second << "\r\n";
   }

   if (_verboseLevel > 3)
   {
      Logger::log() << "Response:" << std::endl
            << "=====================================================" << std::endl
            << responseStream.str() << std::endl;
      if (!response.body.empty()) {
         Logger::log(false)
              << "_____________________________________________________" << std::endl
              << TBLUE(response.body);
      }
      Logger::log(false)
            << "=====================================================" << std::endl
            << std::endl;
   }
   
   // add body
   responseStream
         << "\r\n"
         << response.body;
   std::string responseMsg = responseStream.str();

   send(sock, responseMsg.c_str(), responseMsg.size(), 0);

   return;
}

void HTTPServer::run()
{
    sockaddr_in addrReceive;

    int receiveSocket = socket(AF_INET, SOCK_STREAM, 0);
    if (receiveSocket == -1)
    {
        perror("HTTPServer socket() failed");
        exit(1);
    }

    addrReceive.sin_addr.s_addr = INADDR_ANY;
    addrReceive.sin_port = htons(_port);
    addrReceive.sin_family = AF_INET;

    int mc_reuse = 1;
    if ((setsockopt(receiveSocket, SOL_SOCKET, SO_REUSEADDR, &mc_reuse, sizeof(mc_reuse))) < 0) {
        perror("HTTPServer setsockopt(SO_REUSEADDR) failed");
        exit(1);
    }

    if (bind(receiveSocket, (struct sockaddr*)&addrReceive, sizeof(addrReceive)) == -1)
    {
        perror("HTTPServer bind() failed");
        exit(1);
    }

    if (listen(receiveSocket, 3) == -1)
    {
        perror("HTTPServer listen() failed");
        exit(1);
    }

    while (!isAborted())
    {
       struct timeval tv;
       tv.tv_sec = 1;
       tv.tv_usec = 0;

       socklen_t addr_len = sizeof(addrReceive);

       fd_set fdset;
       FD_ZERO(&fdset);
       FD_SET(receiveSocket, &fdset);
       int retval = select(receiveSocket+1, &fdset, NULL, NULL, &tv);
       if (retval != -1 && FD_ISSET(receiveSocket, &fdset))
       {
           int clientSocket = accept(receiveSocket, (struct sockaddr*)&addrReceive, &addr_len);
           if (clientSocket == -1)
           {
               perror("HTTPServer accept() failed");
               continue;
           }

           handleClient(addrReceive, clientSocket);

           close(clientSocket);
       }
    }

    close(receiveSocket);
}


