/***************************************************************************
 *              SMA Solar Technology AG, 34266 Niestetal, Germany
 ***************************************************************************
 *
 *  Copyright (c) 2008-2014
 *  SMA Solar Technology AG
 *  All rights reserved.
 *  This design is confidential and proprietary of SMA Solar Technology AG.
 *
 ***************************************************************************/

#include <sys/socket.h>
#include <netinet/tcp.h>
#include <stdio.h>
#include <string.h>
#include <sstream>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include "base/Logger.h"
#include "base/PropertyMap.h"
#include "base/StringHelper.h"
#include "HTTPClient.h"

int HTTPResponseHeaderBuffer::getStatusCode() const
{
   int status;
   if (sscanf(buffer.c_str(), "HTTP/%*d.%*d %d", &status) != 1)
      return -1;
   return status;
}

HTTPClient::HTTPClient() :
      _timeout(10)
{
}

int HTTPClient::performRequest(const HTTPRequestHeader &header, const std::string &url, const std::string &body)
{
   // only HTTP supported
   if (!StringHelper::startsWith(url, "http://"))
      return -1;
   std::string urlTmp = url.substr(strlen("http://"));

   // separate BaseURL and URI
   std::string uri = "/";
   size_t uriPos = urlTmp.find('/');
   if (uriPos != std::string::npos)
   {
      uri = urlTmp.substr(uriPos);
      urlTmp = urlTmp.substr(0, uriPos);
   }

   // host (might be IPv4 or [IPv6] address)
   std::string hostIP, hostName;
   sa_family_t addrFamily;
   if (urlTmp.size() > 0 && urlTmp[0] == '[') {
      size_t closeBracketPos = urlTmp.find(']');
      if (closeBracketPos == std::string::npos)
         return -2;
      // host-name (with brackets)
      hostName = urlTmp.substr(0, closeBracketPos + 1);
      // ip (without brackets)
      hostIP = urlTmp.substr(1, closeBracketPos - 1);
      urlTmp = urlTmp.substr(closeBracketPos + 1);
      addrFamily = AF_INET6;
   } else {
      addrFamily = AF_INET;
   }

   // extract port
   in_port_t port = 80;
   size_t portPos = urlTmp.find_last_of(':');
   if (portPos != std::string::npos)
   {
      port = atoi(urlTmp.substr(portPos + 1).c_str());
   }

   // for IPv4 the string up to ':' is the host, for IPv6 we already extracted the host
   if (addrFamily == AF_INET)
   {
      hostIP = urlTmp.substr(0, portPos);
      hostName = hostIP;
   }

   // prepare request

   std::stringstream request;
   switch (header.method)
   {
   case HTTP_METHOD_GET: {
      request
         << "GET " << uri << " HTTP/1.1\r\n"
         << "Host: " << hostName << "\r\n"
         << "\r\n";
      break;
   }
   case HTTP_METHOD_POST: {
      propertyMap_t::const_iterator contTypeIt = header.fields.find("Content-Type");
      std::string contentType = (contTypeIt != header.fields.end() ?
            contTypeIt->second :
            "application/x-www-form-urlencoded");
      request
         << "POST " << uri << " HTTP/1.1\r\n"
         << "Host: " << hostName << "\r\n"
         << "Content-Type: " << contentType << "\r\n"
         << "Content-Length: " << body.size() << "\r\n"
         << "\r\n"
         << body;
      break;
   }
   default:
      return -3;
   }

   struct sockaddr_storage dstAddr;
   socklen_t dstAddrSize;

   if (addrFamily == AF_INET6) {
      // IPv6 address
      struct sockaddr_in6 *dstAddr_v6 = (struct sockaddr_in6 *)&dstAddr;
      dstAddrSize = sizeof(struct sockaddr_in6);
      memset(&dstAddr, 0, dstAddrSize);
      dstAddr_v6->sin6_family = AF_INET6;
      if (inet_pton(AF_INET6, hostIP.c_str(), &dstAddr_v6->sin6_addr) != 1)
         return -4;
      dstAddr_v6->sin6_port = htons(port);
   } else {
      // IPv4 address
      struct sockaddr_in *dstAddr_v4 = (struct sockaddr_in *)&dstAddr;
      dstAddrSize = sizeof(struct sockaddr_in);
      memset(&dstAddr, 0, dstAddrSize);
      dstAddr_v4->sin_family = AF_INET;
      if (inet_pton(AF_INET, hostIP.c_str(), &dstAddr_v4->sin_addr) != 1)
         return -5;
      dstAddr_v4->sin_port = htons(port);
   }

   int sock = openSocket((struct sockaddr*)&dstAddr, dstAddrSize);
   if (sock < 0)
      return -6;

   int result = 0;
   if (!sendRequest(sock, request.str())) {
      result = -7;
   } else if (!HTTPCommon::recvMessage(sock, true, _header.buffer, _body, _timeout)) {
      result = -8;
   }

   close(sock);
   return result;
}

int HTTPClient::openSocket(const struct sockaddr *dstAddr, socklen_t dstAddrSize)
{
   int sock = socket(dstAddr->sa_family, SOCK_STREAM, 0);
   if (sock < 0)
   {
      Logger::log() << "Failed to open socket: " << strerror(errno) << std::endl;
      return -1;
   }

   if (connect(sock, dstAddr, dstAddrSize) < 0)
   {
      Logger::log() << "Failed to connect socket: " << strerror(errno) << std::endl;
      close(sock);
      return -1;
   }

   return sock;
}

bool HTTPClient::sendRequest(int sock, const std::string &request)
{
   if (send(sock, request.c_str(), request.size(), 0) != (ssize_t)request.size())
   {
       Logger::log() << "Failed to send request: " << strerror(errno) << std::endl;
       return false;
   }
   return true;
}

void HTTPClient::reset()
{
   _header.buffer = "";
   _body = "";
}
