/***************************************************************************
 *              SMA Solar Technology AG, 34266 Niestetal, Germany
 ***************************************************************************
 *
 *  Copyright (c) 2008-2014
 *  SMA Solar Technology AG
 *  All rights reserved.
 *  This design is confidential and proprietary of SMA Solar Technology AG.
 *
 ***************************************************************************/

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include "MulticastSocket.h"
#include "Time.h"
#include "Logger.h"

/*
 * TTL values:
 *  0: same host
 *  1: same subnet (won't be forwarded by a router)
 *  <32: same site
 * The UPnP Specification v1.0 suggests a TTL of 2
 */
#define MULTICAST_TTL_DEFAULT 2

MulticastSocket::MulticastSocket(std::string address, int port, bool unicastRecv) :
   _listening(false),
   _listener(0),
   _address(address),
   _sock(-1),
   _port(port),
   _unicastRecv(unicastRecv)
{
   unsigned char mc_ttl = MULTICAST_TTL_DEFAULT;
   int mc_loop = 1;
   int mc_reuse = 1;

   /* create socket to join multicast group on */
   if ((_sock = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
       perror("socket() failed");
       return;
   }

   _interfaces = enumerateInterfaces();

   if ((setsockopt(_sock, IPPROTO_IP, IP_MULTICAST_LOOP, &mc_loop, sizeof(mc_loop))) < 0) {
       perror("setsockopt(IP_MULTICAST_LOOP) failed");
       return;
   }

   if ((setsockopt(_sock, IPPROTO_IP, IP_MULTICAST_TTL, (void*) &mc_ttl, sizeof(mc_ttl))) < 0) {
       perror("setsockopt(IP_MULTICAST_TTL) failed");
       return;
   }

   /* construct a multicast address structure */
   struct sockaddr_in mc_addr;
   memset(&mc_addr, 0, sizeof(mc_addr));
   mc_addr.sin_family      = AF_INET;
   mc_addr.sin_addr.s_addr = htonl(INADDR_ANY);

   // only bind to a specific port if required, otherwise let the OS decide (sin_port = 0)
   if (!_unicastRecv) {
      // set reuse port to on to allow multiple binds per host
      if ((setsockopt(_sock, SOL_SOCKET, SO_REUSEADDR, &mc_reuse, sizeof(mc_reuse))) < 0) {
          perror("setsockopt(SO_REUSEADDR) failed");
          return;
      }

      // bind to given port
      mc_addr.sin_port = htons(_port);
   }

   /* bind address to socket */
   if ((bind(_sock, (struct sockaddr *) &mc_addr, sizeof(mc_addr))) < 0) {
       perror("multicast bind() failed");
       return;
   }
}

MulticastSocket::~MulticastSocket()
{
   abort();
   if (_sock != -1)
      close(_sock);
}

std::vector<network_iface_t> MulticastSocket::enumerateInterfaces()
{
    std::vector<network_iface_t> interfaces;
    struct ifreq ifreqs[10];
    struct ifconf ifconf;

#ifndef LOCALHOST_ONLY
    memset(&ifconf, 0, sizeof(ifconf));
    ifconf.ifc_buf = (char*)ifreqs;
    ifconf.ifc_len = sizeof(ifreqs);

    // retrieve iface names
    int rval = ioctl(_sock, SIOCGIFCONF, (char*) &ifconf);
    if (rval < 0) {
        return interfaces;
    }

    int nifaces =  ifconf.ifc_len / sizeof(struct ifreq);
    for (int i = 0; i < nifaces; i++)
    {
       struct in_addr addr;
       in_addr_t netmask;

       // retrieve iface address
       rval = ioctl(_sock, SIOCGIFADDR, (char*) &ifreqs[i]);
       if (rval < 0)
          continue;
       addr = ((struct sockaddr_in *) &ifreqs[i].ifr_addr)->sin_addr;

       // retrieve iface netmask
       rval = ioctl(_sock, SIOCGIFNETMASK, (char*) &ifreqs[i]);
       if (rval < 0)
          continue;
       netmask = ((struct sockaddr_in *) &ifreqs[i].ifr_netmask)->sin_addr.s_addr;

       interfaces.push_back(network_iface_t(addr, netmask));
    }
#else
    struct in_addr addr;
    in_addr_t netmask;

    addr.s_addr = inet_addr("127.0.0.1");
    netmask = inet_addr("255.0.0.0");
    interfaces.push_back(network_iface_t(addr, netmask));
#endif

    return interfaces;
}

bool MulticastSocket::updateInterfaces()
{
   std::vector<network_iface_t> interfacesNew = enumerateInterfaces();

   bool addressChangedDetected = false;
   if(interfacesNew.size() != _interfaces.size()){
      addressChangedDetected = true;
   }
   if(!addressChangedDetected){
      for(unsigned int a = 0; a < interfacesNew.size(); a++){
         if(interfacesNew[a].addr.s_addr != _interfaces[a].addr.s_addr){
            addressChangedDetected = true;
            break;
         }
      }
   }

   if(addressChangedDetected){
      dropMembership();
      setDataLock();
      _interfaces = interfacesNew;
      unsetDataLock();
      addMembership();
   }

   return addressChangedDetected;
}

//#define USE_IP_MREQN

bool MulticastSocket::addMembership()
{
   // no need to add membership for unicast receive mode
   if (_unicastRecv)
      return true;

   bool success = true;
#ifdef USE_IP_MREQN
   struct ip_mreqn mc_req;
#else
   struct ip_mreq mc_req;
#endif
   mc_req.imr_multiaddr.s_addr = inet_addr(_address.c_str());

   for (unsigned int i = 0; i < _interfaces.size(); i++) {
#ifdef USE_IP_MREQN
      mc_req.imr_address.s_addr = _interfaces[i].addr.s_addr;
      mc_req.imr_ifindex = 0;
#else
      mc_req.imr_interface.s_addr = _interfaces[i].addr.s_addr;
#endif

      /* send an IGMP join request for each interface */
      if ((setsockopt(_sock, IPPROTO_IP, IP_ADD_MEMBERSHIP, (void*) &mc_req, sizeof(mc_req))) < 0) {
         perror("setsockopt(IP_ADD_MEMBERSHIP) failed");
         success = false;
      }
   }

   return success;
}

bool MulticastSocket::dropMembership()
{
   // no need to drop membership for unicast receive mode
   if (_unicastRecv)
      return true;

   bool success = true;
   struct ip_mreq mc_req;
   mc_req.imr_multiaddr.s_addr = inet_addr(_address.c_str());

   for (unsigned int i = 0; i < _interfaces.size(); i++) {
       mc_req.imr_interface.s_addr = _interfaces[i].addr.s_addr;

       /* send a DROP MEMBERSHIP message via setsockopt */
       if ((setsockopt(_sock, IPPROTO_IP, IP_DROP_MEMBERSHIP, (void*) &mc_req, sizeof(mc_req))) < 0) {
          perror("setsockopt(IP_DROP_MEMBERSHIP) failed");
          success = false;
       }
   }
   return success;
}

std::vector<network_iface_t> MulticastSocket::getInterfaces()
{
   std::vector<network_iface_t> result;
   setDataLock();
   result = _interfaces;
   unsetDataLock();
   return result;
}

void MulticastSocket::selectInterface(in_addr_t addr)
{
   struct in_addr ifaddr;
   ifaddr.s_addr = addr;
   // Note:
   // - in some cases with Virtual-Box, bridged mode on and both host and client OS have an IP address assigned,
   // messages are sometimes not sent on the correct device even though getsockopt(IP_MULTICAST_IF) returns the correct
   // address. The multicasts on the bridged device seem to be mirrored then.
   // - sometimes also parallel IGMP joins seem to conflict with the network iface selection.
   if ((setsockopt(_sock, IPPROTO_IP, IP_MULTICAST_IF, &ifaddr, sizeof(ifaddr))) < 0) {
      perror("setsockopt(IP_MULTICAST_IF) failed");
      return;
   }
}

ssize_t MulticastSocket::send(const char* sendBuffer, size_t length)
{
   ssize_t result;

   struct sockaddr_in mc_addr;
   memset(&mc_addr, 0, sizeof(mc_addr));
   mc_addr.sin_family      = AF_INET;
   mc_addr.sin_addr.s_addr = inet_addr(_address.c_str());;
   mc_addr.sin_port        = htons(_port);

   setDataLock();
   result = sendto(_sock, sendBuffer, length, 0, (struct sockaddr*) &mc_addr, sizeof(mc_addr));
   unsetDataLock();

   return result;
}

ssize_t MulticastSocket::send(const char* sendBuffer, size_t length, const struct sockaddr_in &dstAddr)
{
   ssize_t result;

   setDataLock();
   result = sendto(_sock, sendBuffer, length, 0, (const struct sockaddr*) &dstAddr, sizeof(dstAddr));
   unsetDataLock();

   return result;
}

void MulticastSocket::listen(IMulticastListener *listener)
{
   if (!listener || _listening)
      return;
   _listening = true;
   _listener = listener;
   start();
}

void MulticastSocket::run()
{
   struct sockaddr_in from_addr;
   socklen_t from_len;
   char recv_buffer[MAX_LEN];
   ssize_t recv_len;

   timestamp_t lastInterfaceUpdate = Time::getCurrentTime();

   addMembership();

   while (!isAborted()) {
      fd_set fdset;
      FD_ZERO(&fdset);
      FD_SET(_sock, &fdset);

      struct timeval tv;
      tv.tv_sec = 1;
      tv.tv_usec = 0;

      int retval = select(_sock + 1, &fdset, NULL, NULL, &tv);
      if (retval != -1 && FD_ISSET(_sock, &fdset)) {
         /* clear the receive buffers & structs */
         memset(recv_buffer, 0, sizeof(recv_buffer));
         from_len = sizeof(from_addr);
         memset(&from_addr, 0, from_len);

         /* block waiting to receive a packet */
         setDataLock();
         recv_len = recvfrom(_sock, recv_buffer, sizeof(recv_buffer), 0, (struct sockaddr*)&from_addr, &from_len);
         unsetDataLock();
         if (recv_len < 0) {
            perror("recvfrom() failed");
            break;
         }

         if (_listener) {
            _listener->onMulticastReceived(recv_buffer, recv_len, from_addr);
         }
      }

      timestamp_t currTime = Time::getCurrentTime();
      if (currTime - lastInterfaceUpdate > 30) {
         if (updateInterfaces() && _listener) {
            _listener->onInterfaceChanged();
         }
         lastInterfaceUpdate = currTime;
      }
   }

   dropMembership();
}

