/***************************************************************************
 *              SMA Solar Technology AG, 34266 Niestetal, Germany
 ***************************************************************************
 *
 *  Copyright (c) 2008-2014
 *  SMA Solar Technology AG
 *  All rights reserved.
 *  This design is confidential and proprietary of SMA Solar Technology AG.
 *
 ***************************************************************************/

#ifndef ISEMPOBJECT_H_
#define ISEMPOBJECT_H_

#include <set>
#include <string>
#include <sstream>
#include <vector>

namespace SEMP {

class ISEMPObject {
public:
   virtual ~ISEMPObject() {}

   virtual bool isSet(const std::string &field) const = 0;
   virtual void markAsSet(const std::string &field, bool value = true) = 0;
   virtual void serializeXML(std::stringstream &sstream, int level = 0) const = 0;
   virtual bool validateXMLSchema(std::string &errElement) const = 0;
};

class SEMPObject : public ISEMPObject {
public:
   SEMPObject() : _onSetListener(0) {}

   virtual bool isSet(const std::string &field) const {
      return _markedFields.find(field) != _markedFields.end();
   }

   /**
    * Note: if a vector is not empty it is automatically marked as set,
    * empty vectors must be explicitly marked to be serialized
    */
   virtual void markAsSet(const std::string &field, bool value = true) {
      if (value) {
         if (_markedFields.find(field) == _markedFields.end()) {
            _markedFields.insert(field);
            if (_onSetListener)
               _onSetListener->markAsSet(_onSetListenerField);
         }
      } else {
         _markedFields.erase(field);
      }
   }

   void setOnSetListener(ISEMPObject *listener, const std::string &field) {
      _onSetListener = listener;
      _onSetListenerField = field;
   }

   static std::string indentXML(int level) {
      return std::string(level, '\t');
   }

   static std::string escapeString(const std::string &str)
   {
      const std::string specialChars = "<>&\"'";

      std::size_t specialChPos = str.find_first_of(specialChars);
      if (specialChPos == std::string::npos)
         return str;

      std::string result = str;
      do {
         std::string escapeStr;
         const char specialCh = result[specialChPos];
         switch (specialCh) {
         case '<':  escapeStr = "&lt;"; break;
         case '>':  escapeStr = "&gt;"; break;
         case '&':  escapeStr = "&amp;"; break;
         case '"': escapeStr = "&quot;"; break;
         case '\'':  escapeStr = "&apos;"; break;
         }
         result = result.replace(specialChPos, 1, escapeStr);
         specialChPos = result.find_first_of(specialChars, specialChPos + escapeStr.size());
      } while (specialChPos != std::string::npos);
      return result;
   }

   static std::string unescapeString(const std::string &str)
   {
      std::string result = str;

      const std::string escapeSeqs[] = { "&lt;", "&gt;", "&amp;", "&quot;", "&apos;" };
      const char *specialChars = "<>&\"'";

      for (int i = 0; i < 5; ++i) {
         std::size_t escPos = 0;
         while ((escPos = result.find(escapeSeqs[i], escPos)) != std::string::npos) {
            result = result.replace(escPos, escapeSeqs[i].size(), 1, specialChars[i]);
            escPos++;
         }
      }
      return result;
   }

private:
   std::set<std::string> _markedFields;

   /**
    * will be notified when this element is set.
    */
   ISEMPObject *_onSetListener;

   /**
    * field-name passed to markAsSet() when listener is notified
    */
   std::string _onSetListenerField;
};

}

#endif /* ISEMPOBJECT_H_ */
